package net.cscott.pcbmill;

import java.awt.*;
import java.awt.geom.*;

import java.net.MalformedURLException;

import java.util.*;

import org.apache.batik.bridge.BridgeContext;
import org.apache.batik.bridge.BridgeException;
import org.apache.batik.bridge.BridgeExtension;
import org.apache.batik.bridge.GVTBuilder;
import org.apache.batik.bridge.UserAgent;
import org.apache.batik.bridge.UserAgentAdapter;
import org.apache.batik.bridge.ViewBox;

import org.apache.batik.dom.svg.SAXSVGDocumentFactory;
import org.apache.batik.dom.svg.SVGOMDocument;
//import org.apache.batik.dom.svg.DefaultSVGContext; // where is this class?

import org.apache.batik.ext.awt.g2d.DefaultGraphics2D;
import org.apache.batik.ext.awt.g2d.GraphicContext;

import org.apache.batik.gvt.*;
import org.apache.batik.gvt.event.EventDispatcher;
import org.apache.batik.gvt.renderer.*;

import org.apache.batik.util.XMLResourceDescriptor;
import org.apache.batik.util.SVGConstants;

import org.w3c.dom.DOMException;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.svg.SVGAElement;
import org.w3c.dom.svg.SVGAnimatedRect;
import org.w3c.dom.svg.SVGDocument;
import org.w3c.dom.svg.SVGRect;
import org.w3c.dom.svg.SVGSVGElement;
import org.w3c.dom.svg.SVGLength;


import gnu.java.awt.geom.GeneralPath;

/** Read in an SVG document, and parse into a simple shape.
 * Inspiration from <code>org.apache.fop.svg.PDFTranscoder</code>, see
 *  <a href="http://xml.apache.org/fop">http://xml.apache.org/fop</a>.
 * @author C. Scott Ananian <cananian@alumni.princeton.edu>
 */
public class SVGParser {
    // this is what we're building.
    public GeneralPath image = new GeneralPath(GeneralPath.WIND_EVEN_ODD);

    public SVGParser(java.io.File f) throws java.io.IOException {
	this((SVGOMDocument) makeDocumentFactory().createDocument
	     (f.toURL().toString()));
    }
    public SVGParser(java.io.InputStream is) throws java.io.IOException {
	this((SVGOMDocument) makeDocumentFactory().createDocument
	     (SVGConstants.SVG_NAMESPACE_URI,
	      SVGConstants.SVG_SVG_TAG,
	      null /* input.getURI */,
	      is));
    }
    private static SAXSVGDocumentFactory makeDocumentFactory() {
	return new SAXSVGDocumentFactory
	    (XMLResourceDescriptor.getXMLParserClassName());
    }
	
    public SVGParser(SVGOMDocument svgDoc) {
        SVGSVGElement root = svgDoc.getRootElement();
	// get the x, y, width, height, and viewbox attributes.
	SVGLength x = root.getX().getBaseVal();
	SVGLength y = root.getY().getBaseVal();
	SVGLength width = root.getWidth().getBaseVal();
	SVGLength height = root.getHeight().getBaseVal();
	SVGAnimatedRect avb = getViewBox(root); // XXX: root.getViewBox()
	SVGRect vb;
	if (avb!=null) vb = avb.getBaseVal();
	else // if no view box (like for sodipodi), assume 90 dpi
	    // this is based on CSS2 recommendation for 'standard pixel'
	    vb = new SVGRectImpl(convertToInches(x)*90,
				 convertToInches(y)*90,
				 convertToInches(width)*90,
				 convertToInches(height)*90);

        // build the GVT tree
        GVTBuilder builder = new GVTBuilder();
        BridgeContext ctx = new BridgeContext
	    (new SVGParserUserAgent(width, height));
        TextPainter textPainter = null;
        textPainter = new StrokingTextPainter();
        ctx.setTextPainter(textPainter);
        
        GraphicsNode gvtRoot;
        try {
            gvtRoot = builder.build(ctx, svgDoc);
	    //prettyprint(gvtRoot);
        } catch (BridgeException ex) {
            throw new RuntimeException(ex);
        }
        ctx = null;
        builder = null;

	// the url referencing an SVG document may specify viewPort,
	// etc. in the #blah part of the URL.  We don't care about that.

	SVGParserGraphics2D graphics = new SVGParserGraphics2D();
        graphics.setGraphicContext(new org.apache.batik.ext.awt.g2d.GraphicContext());

        gvtRoot.paint(graphics);

	// transform the shape: flip and scale by 1/dpi.
	// scale by convertToInches(width)/vb.getWidth()
	//          convertToInches(height)/vb.getHeight()
	// (scale to the viewbox)
	image.transform(AffineTransform.getScaleInstance
			(convertToInches(width)/vb.getWidth(),
			 -convertToInches(height)/vb.getHeight()));
	// HEAL unclosed paths.
	image = Healer.heal(image);
    }

    /** grab one white-space delimited float from the given reader. */
    private static Float parseOne(java.io.Reader r) {
	try {
	    StringBuffer sb = new StringBuffer();
	    boolean non_ws_seen = false;
	    int c;
	    while (-1 != (c=r.read())) {
		if (Character.isWhitespace((char)c)) {
		    if (non_ws_seen) break;
		} else {
		    non_ws_seen = true;
		    sb.append((char)c);
		}
	    }
	    if (non_ws_seen)
		return Float.valueOf(sb.toString().trim());
	    else
		return null; // EOF.
	} catch (java.io.IOException e) {
	    return null; // error
	}
    }

    /** parse the 'viewBox' attribute of the <svg> tag. */
    private static SVGAnimatedRect getViewBox(SVGSVGElement root) {
	String vb = root.getAttributeNS(null, "viewBox");
	if (vb==null) return null;
	java.io.Reader r = new java.io.StringReader(vb);
	final Float x = parseOne(r);
	if (x==null) return null; // not enough numbers
	final Float y = parseOne(r);
	if (y==null) return null; // not enough numbers
	final Float width = parseOne(r);
	if (width==null) return null; // not enough numbers
	final Float height = parseOne(r);
	if (height==null) return null; // not enough numbers
	if (parseOne(r)!=null) return null; // too many numbers!

	return new SVGAnimatedRect() {
		public SVGRect getAnimVal() {throw new Error("unimplemented");}
		public SVGRect getBaseVal() {
		    return new SVGRectImpl(x.floatValue(), y.floatValue(),
					   width.floatValue(),
					   height.floatValue());
		}
	    };
    }

    /** A helper class to implement <code>SVGRect</code>. */
    private static class SVGRectImpl implements SVGRect {
	final float x,y,width,height;
	SVGRectImpl(float x, float y, float width, float height) {
	    this.x = x;
	    this.y = y;
	    this.width = width;
	    this.height = height;
	}
	SVGRectImpl(double x, double y, double width, double height) {
	    this((float)x, (float)y, (float)width, (float)height);
	}
	public float getHeight() { return height; }
	public float getWidth() { return width; }
	public float getX() { return x; }
	public float getY() { return y; }
	public void setHeight(float height) { throw new Error("read-only"); }
	public void setWidth(float width) { throw new Error("read-only"); }
	public void setX(float x) { throw new Error("read-only"); }
	public void setY(float y) { throw new Error("read-only"); }
	public String toString() {return "["+x+" "+y+" "+width+" "+height+"]";}
    }

    /** Helper method to convert an <code>SVGLength</code> into a
     *  floating-point inches measurement. */
    private static double convertToInches(SVGLength l) {
	double val = l.getValueInSpecifiedUnits();
	if (val==0) return 0; // zero of any unit is 0 inches. =)
	switch(l.getUnitType()) {
	case SVGLength.SVG_LENGTHTYPE_IN:
	    return val;
	case SVGLength.SVG_LENGTHTYPE_CM:
	    return val/2.54;
	case SVGLength.SVG_LENGTHTYPE_MM:
	    return val/25.4;
	case SVGLength.SVG_LENGTHTYPE_PT:
	    // "the points used by CSS2 are equal to 1/72th of an inch."
	    return val/72.;
	case SVGLength.SVG_LENGTHTYPE_PC:
	    // "1 pica is equal to 12 points."
	    return 12*val/72.;
	default:
	    System.err.println("UNKNOWN UNIT! "+l.getUnitType()+" / "+val);
	    return 1; // UNKNOWN OR RELATIVE UNITS.
	}
    }

    /** This class provides the 'user agent' information called for
     *  in the SVG spec. */
    class SVGParserUserAgent extends UserAgentAdapter {
	/** Conversion factor for measurements specified in 'pixels'. */
	final private static int DPI=90;
	/** Width and Height of the SVG canvas. */
	final private double width, height;
	SVGParserUserAgent(SVGLength width, SVGLength height) {
	    this.width = convertToUnits(width);
	    this.height = convertToUnits(height);
	}
	public int dpi() { return DPI; }
	public void displayMessage(String message) {
	    System.err.println("MESSAGE: "+message);
	}
	public void showAlert(String message) {
	    System.err.println("ALERT: "+message);
	}
	public float getPixelUnitToMillimeter() {
	    // image is scaled so that 1 unit = 1/1000 inch.
	    // compute (mm/in) / (dpi)
	    return 1/25.4f/dpi();
	}
	// for compatibility.
	public float getPixelToMM() {
	    return getPixelUnitToMillimeter();
	}
	public Dimension2D getViewportSize() {
	    Dimension d = new Dimension();
	    d.setSize(width, height);
	    return d;
	}
	private double convertToUnits(SVGLength l) {
	    return convertToInches(l) * dpi();
	}
    }

    class SVGParserGraphics2D extends DefaultGraphics2D {
	/**
	 * Default constructor
	 */
	SVGParserGraphics2D(){
	    super(true/*textAsShapes*/);
	}

	/**
	 * This constructor supports the create method
	 */
	public SVGParserGraphics2D(SVGParserGraphics2D g){
	    super(g);
	}

	/**
	 * Creates a new <code>Graphics</code> object that is
	 * a copy of this <code>Graphics</code> object.
	 * @return     a new graphics context that is a copy of
	 *             this graphics context.
	 */
	public Graphics create(){
	    return new SVGParserGraphics2D(this);
	}
	
	public void setGraphicContext(GraphicContext c) {
	    gc = c;
	}

	public void draw(Shape s){
	    // just append this to the current path (not connecting!)
	    image.append(s.getPathIterator(gc.getTransform()), false);
	}
	public void fill(Shape s) {
	    // treat the same as 'draw'
	    draw(s);
	}
	public void dispose() {
	    // nothing needs to be done.
	}
    }

    // test code.
    public static void main(String[] args) throws java.io.IOException {
	String parser = XMLResourceDescriptor.getXMLParserClassName();
	SAXSVGDocumentFactory f = new SAXSVGDocumentFactory(parser);
	String uri = new java.io.File(args[0]).toURL().toString();
	SVGOMDocument doc = (SVGOMDocument) f.createDocument(uri);

	new SVGParser(doc);
	// yeah!
    }

    private static void prettyprint(Shape s, AffineTransform at) {
	for (PathIterator pit=s.getPathIterator(at);
	     !pit.isDone(); pit.next()) {
	    double[] segment = new double[6];
	    int type = pit.currentSegment(segment);
	    System.err.print(" ");
	    int n = 2;
	    switch (type) {
	    case PathIterator.SEG_MOVETO:
		System.err.print("MOVETO: "); break;
	    case PathIterator.SEG_CLOSE:
		System.err.print("CLOSE:  "); break;
	    case PathIterator.SEG_LINETO:
		System.err.print("LINETO: "); break;
	    case PathIterator.SEG_QUADTO: n=4;
		System.err.print("QUADTO: "); break;
	    case PathIterator.SEG_CUBICTO: n=6;
		System.err.print("CUBICTO:"); break;
	    default: n=6;
		System.err.print("UNKNOWN:"); break;
	    }
	    for (int i=0; i<n; i++)
		System.err.print(segment[i]+" ");
	    System.err.println();
	}
    }
    static void prettyprint(GraphicsNode gvtRoot) {
	GVTTreeWalker gvttw = new GVTTreeWalker(gvtRoot);
	prettyprint(gvttw, 0);
    }
    private static void prettyprint(GVTTreeWalker gvttw, int indent) {
	GraphicsNode gn = gvttw.getCurrentGraphicsNode();
	for (int i=0; i<indent; i++)
	    System.out.print(' ');
	System.out.println(gn.toString()+" "+gn.getBounds());
	for (GraphicsNode gnc = gvttw.firstChild(); gnc!=null; gnc=gvttw.getNextSibling())
	    prettyprint(gvttw, indent+1);
	gvttw.setCurrentGraphicsNode(gn);
    }
}
