package net.cscott.mallard;

import java.awt.Color;
import java.awt.color.ColorSpace;

/** <code>LuvColorSpace</code> implements a <code>ColorSpace</code> for the
 *  L*u*v* system.  L*u*v* is more 
 * <a href="http://www.poynton.com/notes/colour_and_gamma/ColorFAQ.html#RTFToC35">perceptually-uniform</a>
 * than the standard sRGB or CIE XYZ color spaces. */
public class LuvColorSpace extends BaseColorSpace {
    private static final LuvColorSpace _CS_Luv = new LuvColorSpace();
    private LuvColorSpace() {
	super(TYPE_Luv, 3);
    }
    public static LuvColorSpace getInstance() { return _CS_Luv; }

    private static final double uprime_D50, vprime_D50;
    static {
	uprime_D50 = 4.*CIE_D50[X] / (CIE_D50[X]+15.*CIE_D50[Y]+3.*CIE_D50[Z]);
	vprime_D50 = 9.*CIE_D50[Y] / (CIE_D50[X]+15.*CIE_D50[Y]+3.*CIE_D50[Z]);
    }
    // CIEXYZ values range from 0 to 1 + 32767/32768. (i.e. just under 2)
    public double[] fromCIEXYZ(double[] color) { // my extension
	double dividend = color[X]+15.*color[Y]+3.*color[Z];
	double uprime = (dividend==0) ? 0 : 4.*color[X] / dividend;
	double vprime = (dividend==0) ? 0 : 9.*color[Y] / dividend;

	double y = color[Y]/CIE_D50[Y];
	double Lstar = (y>MAGIC) ? 116*Math.pow(y,ONE_THIRD)-16 : 903.3*y;
	double ustar = 13*Lstar*(uprime-uprime_D50);
	double vstar = 13*Lstar*(vprime-vprime_D50);
	return new double[] { Lstar, ustar, vstar };
    }
    private static final int L=0, u=1, v=2;
    public double[] toCIEXYZ(double[] color) { // my extension
	// invert the above.
	double nY = (color[L]<8) ? color[L]/903.3 :
	    Math.pow((color[L]+16.)/116., 3)*CIE_D50[Y];
	double uprime = (color[u]/(13.*color[L]))+uprime_D50;
	double vprime = (color[v]/(13.*color[L]))+vprime_D50;
	if (color[L]==0) uprime=vprime=0;
	// 15.*color[Y] + 3.*color[Z] = (4/uprime-1)*color[X]
	// (15.-9/vprime)*color[Y]+3.*color[Z] = -color[X]
	//  ... algebra follows ...
	// color[Y]*(3-5*vprime-(3/4)*uprime)/vprime=color[Z]
	double nZ = nY*(3-(5*vprime)-(3*uprime/4))/vprime;
	double nX = ((9/vprime)-15)*nY - 3*nZ;
	if (color[L]==0) nZ=nX=0;
	if (nZ<0) nZ=0;
	if (nX<0) nX=0;
	return new double[] { nX, nY, nZ };
    }
    public float getMinValue(int component) {
	switch(component) {
	case 0: return 0.0f; // L
	case 1: return -128.0f; // u empirically: -96
	case 2: return -145.0f; // v empirically: -145, ostensibly -128
	default: throw new IllegalArgumentException("Bad component index: "+
						    component);
	}
    }
    public float getMaxValue(int component) {
	switch(component) {
	case 0: return 100.0f; // L
	case 1: return 168.0f; // u empirically: 168, ostensibly 127
	case 2: return 127.0f; // v empirically: 85
	default: throw new IllegalArgumentException("Bad component index: "+
						    component);
	}
    }
    public String getName(int component) {
	switch(component) {
	case 0: return "L*"; // CIE Lightness
	case 1: return "u*";
	case 2: return "v*";
	default: throw new IllegalArgumentException("Bad component index: "+
						    component);
	}
    }
    /** Self-test. */
    public static void main(String[] args) {
	ColorSpace rgb = ColorSpace.getInstance(ColorSpace.CS_sRGB);
	ColorSpace xyz = ColorSpace.getInstance(ColorSpace.CS_CIEXYZ);
	LuvColorSpace luv = LuvColorSpace.getInstance();
	double[] min=new double[]{Double.POSITIVE_INFINITY,Double.POSITIVE_INFINITY,Double.POSITIVE_INFINITY};
	double[] max=new double[]{Double.NEGATIVE_INFINITY,Double.NEGATIVE_INFINITY,Double.NEGATIVE_INFINITY};
	final int DIVS=32;
	for (int r=0; r<DIVS; r++) {
	    System.out.print('.');
	    for (int g=0; g<DIVS; g++)
		for (int b=0; b<DIVS; b++) {
		    // XXX: work around bug in java.awt.Color that makes it
		    // break when used with ColorSpaces where the components
		    // range below 0.0 or above 1.0.
		    float[] rgb0 = new float[] {
			(float)r/(DIVS-1),
			(float)g/(DIVS-1),
			(float)b/(DIVS-1) };
		    /*
		    if (r==1) rgb0[0]=1f/255;
		    if (g==1) rgb0[1]=1f/255;
		    if (b==1) rgb0[2]=1f/255;
		    */
		    double[] luv1 = luv.fromRGB(f2d(rgb0));
		    float[] rgb2 = d2f(luv.toRGB(luv1));
		    for (int i=0; i<3; i++) {
			min[i]=Math.min(min[i],luv1[i]);
			max[i]=Math.max(max[i],luv1[i]);
		    }
		    Color cr0 = new Color(rgb0[0],rgb0[1],rgb0[2]);
		    Color cr2 = new Color(rgb2[0],rgb2[1],rgb2[2]);
		    assert cr0.equals(cr2) : cr0+" -> "+cr2;
		    /*
		    System.out.println(f2s(cie0));
		    System.out.println("\t\t->"+d2s(luv1)+"->");
		    System.out.println(f2s(cie2));
		    */
		    //System.out.println(f2s(rgb0)+"->"+f2s(cie0)+"->"+
		    //		       f2s(luv1)+"->"+f2s(cie2)+"->"+
		    //		       f2s(rgb2));

		}
	}
	System.out.println();
	System.out.println("MIN: "+d2s(min));
	System.out.println("MAX: "+d2s(max));
    }
}
