package net.cscott.mallard;

import java.awt.Color;
import java.awt.color.ColorSpace;

/** <code>LuvColorSpace</code> implements a <code>ColorSpace</code> for the
 *  L*a*b* system.  L*a*b* is more 
 * <a href="http://www.poynton.com/notes/colour_and_gamma/ColorFAQ.html#RTFToC35">perceptually-uniform</a>
 * than the standard sRGB or CIE XYZ color spaces. */
public class LabColorSpace extends BaseColorSpace {
    private static final LabColorSpace _CS_Lab = new LabColorSpace();
    private LabColorSpace() {
	super(TYPE_Lab, 3);
    }
    public static LabColorSpace getInstance() { return _CS_Lab; }

    // CIEXYZ values range from 0 to 1 + 32767/32768. (i.e. just under 2)
    // L* = 116*((y/yn)^(1/3))-16 (limited to zero)
    //   strictly speaking, value only for y/yn > .008856
    // a* =500*((x/xn)^1/3 - (y/yn)^1/3)
    // b* = 200*((y/yn)^1/3 - (z/zn)^1/3)
    public double[] fromCIEXYZ(double[] color) {
	double x = color[X]/CIE_D50[X];
	double y = color[Y]/CIE_D50[Y];
	double z = color[Z]/CIE_D50[Z];
	double m_cbrt_x = mod_cbrt(x);
	double m_cbrt_y = mod_cbrt(y);
	double m_cbrt_z = mod_cbrt(z);
	double Lstar = (y>MAGIC) ? 116*m_cbrt_y-16 : 903.3*y;
	double astar = 500.*(m_cbrt_x - m_cbrt_y);
	double bstar = 200.*(m_cbrt_y - m_cbrt_z);
	return new double[] { Lstar, astar, bstar };
    }
    private static double mod_cbrt(double t) {
	return (t>MAGIC) ? Math.pow(t,ONE_THIRD) : 7.787*t + (16/116.);
    }
    private static double mod_cube(double t) {
	return (t>CBRT_MAGIC) ? t*t*t : (t-(16/116.))/7.787;
    }

    private static final int L=0, a=1, b=2;
    public double[] toCIEXYZ(double[] color) {
	// invert the above.
	double y, m_cbrt_y;
	if (color[L]<8) {
	    y = color[L]/903.3;
	    m_cbrt_y = mod_cbrt(y);
	} else {
	    m_cbrt_y = (color[L]+16.)/116.;
	    y = m_cbrt_y*m_cbrt_y*m_cbrt_y;
	}
	double m_cbrt_x = (color[a]/500.)+m_cbrt_y;
	double m_cbrt_z = (-color[b]/200.)+m_cbrt_y;
	double x = mod_cube(m_cbrt_x);
	double z = mod_cube(m_cbrt_z);
	return new double[] { x*CIE_D50[X], y*CIE_D50[Y], z*CIE_D50[Z] };
    }
    public float getMinValue(int component) {
	switch(component) {
	case 0: return 0.0f; // L
	case 1: return -128.0f; // a empirically: -88
	case 2: return -128.0f; // b empirically: -127
	default: throw new IllegalArgumentException("Bad component index: "+
						    component);
	}
    }
    public float getMaxValue(int component) {
	switch(component) {
	case 0: return 100.0f; // L
	case 1: return 127.0f; // a empirically: 97
	case 2: return 127.0f; // b empirically: 85
	default: throw new IllegalArgumentException("Bad component index: "+
						    component);
	}
    }
    public String getName(int component) {
	switch(component) {
	case 0: return "L*"; // CIE Lightness
	case 1: return "a*";
	case 2: return "b*";
	default: throw new IllegalArgumentException("Bad component index: "+
						    component);
	}
    }
    /** Self-test. */
    public static void main(String[] args) {
	ColorSpace rgb = ColorSpace.getInstance(ColorSpace.CS_sRGB);
	ColorSpace xyz = ColorSpace.getInstance(ColorSpace.CS_CIEXYZ);
	LabColorSpace lab = LabColorSpace.getInstance();
	double[] min=new double[]{Double.POSITIVE_INFINITY,Double.POSITIVE_INFINITY,Double.POSITIVE_INFINITY};
	double[] max=new double[]{Double.NEGATIVE_INFINITY,Double.NEGATIVE_INFINITY,Double.NEGATIVE_INFINITY};
	final int DIVS=32;
	for (int r=0; r<DIVS; r++)
	    for (int g=0; g<DIVS; g++)
		for (int b=0; b<DIVS; b++) {
		    // XXX: work around bug in java.awt.Color that makes it
		    // break when used with ColorSpaces where the components
		    // range below 0.0 or above 1.0.
		    float[] rgb0 = new float[] {
			(float)r/(DIVS-1),
			(float)g/(DIVS-1),
			(float)b/(DIVS-1) };
		    double[] lab1 = lab.fromRGB(f2d(rgb0));
		    float[] rgb2 = d2f(lab.toRGB(lab1));
		    for (int i=0; i<3; i++) {
			min[i]=Math.min(min[i],lab1[i]);
			max[i]=Math.max(max[i],lab1[i]);
		    }
		    Color cr0 = new Color(rgb0[0],rgb0[1],rgb0[2]);
		    Color cr2 = new Color(rgb2[0],rgb2[1],rgb2[2]);
		    assert cr0.equals(cr2) : cr0+" -> "+cr2;
		    /*
		    System.out.println(f2s(cie0)+"->"+f2s(lab1)+"->"+f2s(cie2));
		    */
		    /*
		    System.out.println(f2s(rgb0)+"->"+f2s(cie0)+"->"+d2s(lab1)+
				       "->"+f2s(cie2)+"->"+f2s(rgb2));
		    */
		}
	System.out.println();
	System.out.println("MIN: "+d2s(min));
	System.out.println("MAX: "+d2s(max));
    }
}
