package net.cscott.pcbmill;

import java.awt.*;
import java.awt.geom.*;
import java.util.List;
import java.util.*;

import gnu.java.awt.geom.GeneralPath;

public abstract class Healer {
    public static GeneralPath heal(Shape gp) {
	GeneralPath result = new GeneralPath
	    (gp.getPathIterator(null).getWindingRule());

	List<GeneralPath> parts = split(gp);
	for (Iterator<GeneralPath> it=parts.iterator(); it.hasNext(); ) {
	    GeneralPath a_part = it.next();
	    if (!isClosed(a_part)) continue;
	    result.append(a_part, false);
	    it.remove();
	}

	// if all shapes are closed, we're done!
	if (parts.isEmpty()) return result;

	// create a grid data structure, appropriately sized.
	Grid<GeneralPath> g=new Grid<GeneralPath>(bbox(parts), parts.size()*2);
	// add the endpoints of all the open simple paths to it.
	for (Iterator<GeneralPath> it=parts.iterator(); it.hasNext(); ) {
	    GeneralPath a_part = it.next();
	    List<Point2D> endpoints = endpoints(a_part);
	    assert endpoints.size()==2;
	    g.add(endpoints.get(0), a_part);
	    g.add(endpoints.get(1), a_part);
	}
	// pick first path part arbitrarily.
	GeneralPath first_part = g.removeOne();
	result.append(first_part, false);
	List<Point2D> first_endpoints = endpoints(first_part);
	Point2D start_endpoint = first_endpoints.get(0);
	Point2D last_endpoint = first_endpoints.get(1);
	// now, while there are points left in the grid:
	while (!g.isEmpty()) {
	    assert last_endpoint != null;
	    assert start_endpoint != null;
	    // find closest part to last endpoint.
	    GeneralPath a_part = g.removeClosestTo(last_endpoint);
	    // maybe our start point is closer?
	    List<Point2D> endpoints = endpoints(a_part);
	    double dist_start = last_endpoint.distance(endpoints.get(0));
	    double dist_end   = last_endpoint.distance(endpoints.get(1));
	    double dist_close = last_endpoint.distance(start_endpoint);
	    if (dist_close <= dist_start && dist_close <= dist_end) {
		// start point is closer: close this path.
		result.closePath();
		// start a new one.
		result.append(a_part, false);
		start_endpoint = endpoints.get(0);
		last_endpoint = endpoints.get(1);
	    } else {
		// new path is closer: append to the last path.
		if (dist_start > dist_end) {
		    // oops, reverse this bit so that we connect to its end
		    a_part = reversePath(a_part);
		    last_endpoint = endpoints.get(0);
		} else
		    last_endpoint = endpoints.get(1);
		result.append(a_part, true);
	    }
	}
	// done.  close last path (whether it needs it or not) and return!
	result.closePath();
	return result;
    }

    /** split all subpaths in s into their own paths. */
    private static List<GeneralPath> split(Shape s) {
	List<GeneralPath> result = new ArrayList<GeneralPath>();
	GeneralPath current = null;
	boolean seen_non_move = false, seen_close = false;;
	
	double[] segment = new double[6];
	for (PathIterator pi=s.getPathIterator(null); !pi.isDone(); pi.next()){
	    switch(pi.currentSegment(segment)) {
	    case PathIterator.SEG_MOVETO:
		// start new path.
		if (current!=null && seen_non_move) result.add(current);
		current = new GeneralPath(pi.getWindingRule());
		current.moveTo(segment[0], segment[1]);
		seen_close = false;  seen_non_move = false;
		break;
	    case PathIterator.SEG_CLOSE:
		assert seen_non_move;
		if (!seen_close)
		    current.closePath();
		seen_close = true;
		break;
	    case PathIterator.SEG_LINETO:
		seen_non_move = true;
		current.lineTo(segment[0], segment[1]);
		break;
	    case PathIterator.SEG_QUADTO:
		seen_non_move = true;
		current.quadTo(segment[0], segment[1], segment[2], segment[3]);
		break;
	    case PathIterator.SEG_CUBICTO:
		seen_non_move = true;
		current.curveTo(segment[0], segment[1], segment[2],
				segment[3], segment[4], segment[5]);
		break;
	    default:
		assert false : "unknown path segment type";
		break;
	    }
	}
	if (current!=null && seen_non_move) result.add(current);
	return result;
    }
    /** Determine if a simple path in gp is open or closed. */
    private static boolean isClosed(Shape s) {
	double[] segment = new double[6];
	boolean seen_move = false, seen_close = false;
	for (PathIterator pi=s.getPathIterator(null); !pi.isDone(); pi.next()){
	    switch(pi.currentSegment(segment)) {
	    case PathIterator.SEG_MOVETO:
		assert !seen_move: "this path is not simple!";
		seen_move = true;
		break;
	    case PathIterator.SEG_CLOSE:
		assert !seen_close:
		    "simple path may not have more than one close!";
		seen_close = true;
		break;
	    default:
		assert !seen_close: "close must be last in path";
		break;
	    }
	}
	return seen_close;
    }
    /** Determine the bounding box of a list of simple paths. */
    private static Rectangle2D bbox(List<GeneralPath> l) {
	assert l.size() > 0;
	Iterator<GeneralPath> it=l.iterator();
	Rectangle2D result = it.next().getBounds2D();
	while (it.hasNext())
	    result = result.createUnion(it.next().getBounds2D());
	return result;
    }
    /** Return the endpoints of a simple path. */
    private static List<Point2D> endpoints(Shape s) {
	List<Point2D> result = new ArrayList<Point2D>(2);

	double[] segment = new double[6];
	boolean seen_move = false;
	double last_x=0, last_y=0;
	for (PathIterator pi=s.getPathIterator(null); !pi.isDone(); pi.next()){
	    switch(pi.currentSegment(segment)) {
	    case PathIterator.SEG_MOVETO:
		assert !seen_move: "this path is not simple!";
		seen_move = true;
		last_x = segment[0]; last_y = segment[1];
		result.add(new Point2D.Double(last_x, last_y));
		break;
	    case PathIterator.SEG_CLOSE:
		assert false : "this path ought to be open!";
		break;
	    case PathIterator.SEG_LINETO:
		last_x = segment[0]; last_y = segment[1];
		break;
	    case PathIterator.SEG_QUADTO:
		last_x = segment[2]; last_y = segment[3];
		break;
	    case PathIterator.SEG_CUBICTO:
		last_x = segment[4]; last_y = segment[5];
		break;
	    default:
		assert false : "unknown path segment";
		break;
	    }
	}
	assert seen_move : "no start seen on path";
	result.add(new Point2D.Double(last_x, last_y));
	assert result.size()==2;
	return result;
    }
    /** Reverse the given path so that it goes from its end to its start. */
    private static GeneralPath reversePath(Shape s) {
	// determine the number of segments in the path.
	int size=0;
	for (PathIterator pi=s.getPathIterator(null); !pi.isDone(); pi.next())
	    size++;

	// create space to hold all the path segments.
	int[] types = new int[size];
	double[] segments = new double[8*size];

	// copy all the segments into our array, in reverse order.
	double last_x=0, last_y=0, start_x=0, start_y=0;
	double[] segment = new double[6];
	for (PathIterator pi=s.getPathIterator(null); !pi.isDone(); pi.next()){
	    types[--size] = pi.currentSegment(segment);
	    segments[8*size + 0] = last_x;
	    segments[8*size + 1] = last_y;
	    System.arraycopy(segment, 0, segments, 2+8*size, 6);
	    switch(types[size]) {
	    case PathIterator.SEG_MOVETO:
		last_x = segment[0]; last_y = segment[1];
		start_x = segment[0]; start_y = segment[1];
		break;
	    case PathIterator.SEG_CLOSE:
		segment[8*size + 2] = start_x;
		segment[8*size + 3] = start_y;
		last_x = start_x; last_y = start_y;
		break;
	    case PathIterator.SEG_LINETO:
		last_x = segment[0]; last_y = segment[1];
		break;
	    case PathIterator.SEG_QUADTO:
		last_x = segment[2]; last_y = segment[3];
		break;
	    case PathIterator.SEG_CUBICTO:
		last_x = segment[4]; last_y = segment[5];
		break;
	    default:
		assert false : "unknown path segment";
		break;
	    }
	}

	// now go through the array, making the reversed path.
	GeneralPath result = new GeneralPath
	    (s.getPathIterator(null).getWindingRule());
	boolean move_done = false, should_close = false;
	for (int i=0; i<types.length; i++) {
	    switch(types[i]) {
	    case PathIterator.SEG_MOVETO:
		if (should_close) result.closePath();
		should_close = false;
		move_done = false;
		break;
	    case PathIterator.SEG_CLOSE:
		should_close = true;
		break;
	    case PathIterator.SEG_LINETO:
		if (!move_done) {
		    result.moveTo(segments[8*i+2], segments[8*i+3]);
		    move_done = true;
		}
		result.lineTo(segments[8*i+0], segments[8*i+1]);
		break;
	    case PathIterator.SEG_QUADTO:
		if (!move_done) {
		    result.moveTo(segments[8*i+4], segments[8*i+5]);
		    move_done = true;
		}
		result.quadTo(segments[8*i+2], segments[8*i+3],
			      segments[8*i+0], segments[8*i+1]);
		break;
	    case PathIterator.SEG_CUBICTO:
		if (!move_done) {
		    result.moveTo(segments[8*i+6], segments[8*i+7]);
		    move_done = true;
		}
		result.curveTo(segments[8*i+4], segments[8*i+5],
			       segments[8*i+2], segments[8*i+3],
			       segments[8*i+0], segments[8*i+1]);
		break;
	    default:
		assert false : "unknown path segment";
		break;
	    }
	}
	// okay, done!
	return result;
    }

    public static class Grid<T> {
	/** grid size */
	final int m;
	/** grid offset */
	final double minX, minY;
	/** grid spacing */
	final double grid_width, grid_height;
	/** grid nodes */
	final List<List<Node>> nodes;
	/** items in the grid */
	int size = 0;
	/** Reverse map of values to node locations. */
	final Map<T, Node> where = new LinkedHashMap<T, Node>();
	Grid(Rectangle2D bounds, int num_points) {
	    // grid size m
	    this.m = (int)Math.ceil(Math.sqrt(num_points));
	    // new (m x m) array
	    this.nodes = new ArrayList<List<Node>>(m);
	    for (int i=0; i<m; i++)
		nodes.add(i,new ArrayList<Node>(Collections.<Node>nCopies(m, null)));
	    this.minX = bounds.getMinX();
	    this.minY = bounds.getMinY();
	    this.grid_width = bounds.getWidth()/m;
	    this.grid_height = bounds.getHeight()/m;
	}

	public boolean isEmpty() {
	    return size==0;
	}
	public void add(Point2D pt, T value) {
	    int i = (int) ((pt.getX()-minX)/grid_width);
	    int j = (int) ((pt.getY()-minY)/grid_height);
	    if (i<0) i=0; if (i>=m) i=m-1;
	    if (j<0) j=0; if (j>=m) j=m-1;
	    nodes.get(i).set(j, new Node(pt, value, nodes.get(i).get(j)));
	}
	public T removeOne() {
	    assert !isEmpty();
	    T t = where.keySet().iterator().next();
	    removeValue(t);
	    return t;
	}
	public T removeClosestTo(Point2D pt) {
	    assert !isEmpty();
	    // spiral search through grid for closest point.
	    int i = (int)((pt.getX()-minX)/grid_width);
	    int j = (int)((pt.getY()-minY)/grid_height);
	    if (i<0) i=0; if (i>=m) i=m-1;
	    if (j<0) j=0; if (j>=m) j=m-1;
	    T closest=null; double dist = 0;
	    for (GridIterator<Node> gi=(size<m) ?
		     (GridIterator<Node>) new NodeLinearIterator(i,j) :
		     (GridIterator<Node>) new NodeSpiralIterator(i, j);
		 gi.hasNext(); ) {
		for (Node n = gi.next(); n!=null; n=n.next) {
		    if (closest==null || pt.distance(n.point) < dist) {
			closest = n.value;
			dist = pt.distance(n.point);
		    }
		}
		if (gi.checkPoint() && closest!=null) {
		    removeValue(closest);
		    return closest;
		}
	    }
	    assert false;
	    return null;
	}

	private static interface GridIterator<X> extends Iterator<X> {
	    public boolean checkPoint();
	}
	private class NodeLinearIterator implements GridIterator<Node> {
	    final Node start; Iterator<Node> rest=null;
	    NodeLinearIterator(int x, int y) {
		this.start = nodes.get(x).get(y);
	    }
	    public boolean hasNext() { return rest==null || rest.hasNext(); }
	    public Node next() {
		if (rest==null) {
		    rest = where.values().iterator();
		    return start;
		} else {
		    Node n = rest.next();
		    return (n==start && rest.hasNext()) ? rest.next() : n;
		}
	    }
	    public boolean checkPoint() { return !hasNext(); }
	    public void remove() { assert false; }
	}
	private class NodeSpiralIterator implements GridIterator<Node> {
	    final SpiralIterator si; boolean check=false;
	    NodeSpiralIterator(int x, int y) {
		this.si = new SpiralIterator(x,y);
	    }
	    public boolean hasNext() { return true; }
	    public Node next() {
		Node n = null;
		for( ; n==null; si.next()) {
		    // don't check out-of-bounds grid square.
		    boolean x_out = si.currentX() < 0 || si.currentX() >= m;
		    boolean y_out = si.currentY() < 0 || si.currentY() >= m;
		    if (x_out && y_out)
			si.skipSide();
		    else if (x_out || y_out)
			/* skip this one */;
		    else n = nodes.get(si.currentX()).get(si.currentY());
		    if (si.lastInSpiral()) check=true;
		}
		return n;
	    }
	    public boolean checkPoint() {
		try { return check; } finally { check=false; }
	    }
	    public void remove() { assert false; }
	}

	private static class SpiralIterator {
	    int size;
	    int x, y;
	    int blocks_left;
	    byte direction;
	    SpiralIterator(int start_x, int start_y) {
		this.x = start_x; this.y = start_y;
		this.size = 0; this.direction=0; this.blocks_left=0;
	    }
	    int currentX() { return x; }
	    int currentY() { return y; }
	    boolean lastInSpiral() { return blocks_left==0 && direction==0; }
	    void skipSide() {
		switch(direction) {
		case 3: x+=blocks_left; break;
		case 2: y+=blocks_left; break;
		case 1: x-=blocks_left; break;
		case 0: y-=blocks_left; break;
		default: assert false;
		}
		blocks_left=0;
	    }
	    void next() {
		if (blocks_left==0) {
		    if (direction==0) {
			size++;
			x--; y--; // move diagonally up one.
			direction=4;
		    }
		    direction--;
		    blocks_left = 2*size;
		}			
		switch(direction) {
		case 3: x++; break;
		case 2: y++; break;
		case 1: x--; break;
		case 0: y--; break;
		default: assert false;
		}
		blocks_left--;
	    }
	}

	private void removeValue(T t) {
	    Node n = where.get(t);
	    assert n!=null : "t is not present in grid!";
	    Node nn=n;
	    do {
		nn.unlink();
		nn = nn.same_value;
	    } while (nn!=n);
	    where.remove(t);
	}

	private class Node {
	    final Point2D point;
	    final T value;
	    Node prev, next;
	    Node same_value;
	    Node(Point2D point, T value, Node next) {
		assert next==null || next.prev==null;
		this.point = point;
		this.value = value;
		this.prev = null;
		this.next = next;
		if (next!=null) next.prev = this;
		// increase size
		size++;
		// add to where
		Node other = where.get(value);
		if (other==null) { // only one with this value.
		    where.put(value, this);
		    this.same_value = this;
		} else { // already another w/ same value.
		    this.same_value = other.same_value;
		    other.same_value = this;
		}
	    }
	    // remove this node from the grid.
	    void unlink() {
		if (this.prev!=null) this.prev.next = this.next;
		if (this.next!=null) this.next.prev = this.prev;
		if (this.prev==null) {
		    // rehash in grid.
		    int i = (int)((point.getX()-minX)/grid_width);
		    int j = (int)((point.getY()-minY)/grid_height);
		    if (i<0) i=0; if (i>=m) i=m-1;
		    if (j<0) j=0; if (j>=m) j=m-1;
		    assert nodes.get(i).get(j) == this;
		    nodes.get(i).set(j, this.next);
		}
		// decrease size.
		size--;
		// don't need to remove from where; this will be done
		// for us.
	    }
	}
    }
}
