/* GeneralPath.java -- represents a shape built from subpaths
   Copyright (C) 2002 Free Software Foundation

This file is part of GNU Classpath.

GNU Classpath is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU Classpath is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under
terms of your choice, provided that you also meet, for each linked
independent module, the terms and conditions of the license of that
module.  An independent module is a module which is not derived from
or based on this library.  If you modify this library, you may extend
this exception to your version of the library, but you are not
obligated to do so.  If you do not wish to do so, delete this
exception statement from your version. */


package gnu.java.awt.geom;
// --- this stuff can be removed once the package is changed back
import java.awt.geom.*;
// --- end stuff to be removed.

import java.awt.Rectangle;
import java.awt.Shape;

/**
 * Note that Sun's implementation only expects float precision, not double.
 * GNU Does Things Better (tm) by maintaining double precision.
 *
 * @author C. Scott Ananian <cananian@alumni.princeton.edu>
 * @author Eric Blake <ebb9@email.byu.edu>
 */
public final class GeneralPath implements Shape, Cloneable
{
  public static final int WIND_EVEN_ODD = PathIterator.WIND_EVEN_ODD;
  public static final int WIND_NON_ZERO = PathIterator.WIND_NON_ZERO;

  /** Initial size if not specified. */
  private static final int INIT_SIZE = 20;

  /** The winding rule. */
  private int rule;
  /**
   * The path type in points. Note that points[index] maps to
   * types[index >> 1]; the control points of quad and cubic paths map as
   * well but are ignored.
   */
  private byte[] types;
  /**
   * The list of all points seen. We keep it in double precision, even though
   * Sun (for some mysterious reason) chose to allow only floats.
   */
  private double[] points;
  /** The index of the most recent moveto point, or null. */
  private int subpath = -1;
  /** The next available index into points. */
  private int index;
  /** Bounding box for the path.  */
  private double bound_min_x, bound_min_y, bound_max_x, bound_max_y;
  /** Whether this bounding box is valid; it may be invalid if there are
   *  no points in the path yet. */
  private boolean bounds_valid = false;

  public GeneralPath()
  {
    this(WIND_NON_ZERO, INIT_SIZE);
  }
  public GeneralPath(int rule)
  {
    this(rule, INIT_SIZE);
  }
  public GeneralPath(int rule, int capacity)
  {
    if (rule != WIND_EVEN_ODD && rule != WIND_NON_ZERO)
      throw new IllegalArgumentException();
    this.rule = rule;
    if (capacity < INIT_SIZE)
      capacity = INIT_SIZE;
    types = new byte[capacity >> 1];
    points = new double[capacity];
  }
  public GeneralPath(Shape s)
  {
    types = new byte[INIT_SIZE >> 1];
    points = new double[INIT_SIZE];
    PathIterator pi = s.getPathIterator(null);
    setWindingRule(pi.getWindingRule());
    append(pi, false);
  }

  public void moveTo(float x, float y)
  {
      moveTo((double)x, (double)y);
  }
  /** GNU extension. */
  public void moveTo(double x, double y)
  {
    subpath = index;
    ensureSize(index + 2);
    types[index >> 1] = PathIterator.SEG_MOVETO;
    points[index++] = x;
    points[index++] = y;
    updateBounds(x, y);
  }
  public void lineTo(float x, float y)
  {
      lineTo((double)x, (double)y);
  }
  /** GNU extension. */
  public void lineTo(double x, double y)
  {
    ensureSize(index + 2);
    types[index >> 1] = PathIterator.SEG_LINETO;
    points[index++] = x;
    points[index++] = y;
    updateBounds(x, y);
  }
  public void quadTo(float x1, float y1, float x2, float y2) {
    quadTo((double)x1,(double)y1,(double)x2,(double)y2);
  }
  /** GNU extension. */
  public void quadTo(double x1, double y1, double x2, double y2)
  {
    ensureSize(index + 4);
    types[index >> 1] = PathIterator.SEG_QUADTO;
    points[index++] = x1;
    points[index++] = y1;
    points[index++] = x2;
    points[index++] = y2;
    updateBounds(x1, y1);
    updateBounds(x2, y2);
  }
  public void curveTo(float x1, float y1, float x2, float y2,
                      float x3, float y3) {
    curveTo((double)x1,(double)y1,(double)x2,(double)y2,
	    (double)x3,(double)y3);
  }
  /** GNU extension. */
  public void curveTo(double x1, double y1, double x2, double y2,
                      double x3, double y3)
  {
    ensureSize(index + 6);
    types[index >> 1] = PathIterator.SEG_CUBICTO;
    points[index++] = x1;
    points[index++] = y1;
    points[index++] = x2;
    points[index++] = y2;
    points[index++] = x3;
    points[index++] = y3;
    updateBounds(x1, y1);
    updateBounds(x2, y2);
    updateBounds(x3, y3);
  }
  public void closePath()
  {
    ensureSize(index + 2);
    types[index >> 1] = PathIterator.SEG_CLOSE;
    points[index++] = points[subpath];
    points[index++] = points[subpath + 1];
    // doesn't affect bounding box.
  }

  public void append(Shape s, boolean connect)
  {
    append(s.getPathIterator(null), connect);
  }
  public void append(PathIterator i, boolean connect)
  {
    double[] f = new double[6];
    for ( ; ! i.isDone(); i.next())
      {
        int result = i.currentSegment(f);
        switch (result)
          {
          case PathIterator.SEG_MOVETO:
            if (! connect)
              {
                moveTo(f[0], f[1]);
                break;
              }
	    connect=false;
            if (subpath >= 0 && f[0] == points[index - 2]
                && f[1] == points[index - 1])
              break; // skip point; we already connect.
            // Fallthrough.
          case PathIterator.SEG_LINETO:
            lineTo(f[0], f[1]);
            break;
          case PathIterator.SEG_QUADTO:
            quadTo(f[0], f[1], f[2], f[3]);
            break;
          case PathIterator.SEG_CUBICTO:
            curveTo(f[0], f[1], f[2], f[3], f[4], f[5]);
            break;
	  case PathIterator.SEG_CLOSE:
            closePath();
	    break;
          default:
	    assert false : i;
          }
      }
  }

  public int getWindingRule()
  {
    return rule;
  }
  public void setWindingRule(int rule)
  {
    if (rule != WIND_EVEN_ODD && rule != WIND_NON_ZERO)
      throw new IllegalArgumentException();
    this.rule = rule;
  }

  public Point2D getCurrentPoint()
  {
    if (index < 2)
      return null;
    return new Point2D.Double(points[index - 2], points[index - 1]);
  }
  public void reset()
  {
    subpath = -1;
    index = 0;
    bounds_valid = false;
  }

  public void transform(AffineTransform xform)
  {
    xform.transform(points, 0, points, 0, index >> 1);
    // recompute bounds.
    if (index<2) bounds_valid=false;
    else {
      bounds_valid=true;
      bound_min_x = bound_max_x = points[0];
      bound_min_y = bound_max_y = points[1];
      for (int i=2; i<index; i+=2)
	updateBounds(points[i], points[i+1]);
    }
  }
  public Shape createTransformedShape(AffineTransform xform)
  {
    GeneralPath p = new GeneralPath(this);
    p.transform(xform);
    return p;
  }

  public Rectangle getBounds()
  {
    return getBounds2D().getBounds();
  }
  public Rectangle2D getBounds2D()
  {
    if (bounds_valid)
      return new Rectangle2D.Double(bound_min_x, bound_min_y,
				    bound_max_x-bound_min_x,
				    bound_max_y-bound_min_y);
    // no points in this path!
    else return new Rectangle2D.Double(0,0,0,0);
  }
  private void updateBounds(double x, double y) {
    // bezier curves are contained within the convex hull of their
    // control points.  So we can estimate the bounds by computing
    // the bounding box of all the control points.
    if (bounds_valid) {
      if (x<bound_min_x) bound_min_x=x;
      if (y<bound_min_y) bound_min_y=y;
      if (x>bound_max_x) bound_max_x=x;
      if (y>bound_max_y) bound_max_y=y;
    } else {
      bound_min_x=bound_max_x=x;
      bound_min_y=bound_max_y=y;
      bounds_valid=true;
    }
  }

  public boolean contains(double x, double y)
  {
    if (!getBounds2D().contains(x,y)) return false; // fast path.
    // following "CROSSINGS" in
    //  Haines, Eric.  "Point in Polygon Strategies," Graphics Gems IV.
    //  ed Paul Heckbert, Academic Press, p 24-46, 1994.
    // an early version of this is available at
    //  http://www.acm.org/pubs/tog/editors/erich/ptinpoly/
    int crossings = 0;  boolean inside_flag = false;
    
    Point2D last = new Point2D.Double(), start = new Point2D.Double();
    double[] segment = new double[6];
    boolean open = false; int curSeg;
    for (PathIterator pi=getPathIterator(null); !pi.isDone()||open; ){
      // some shenanigans at the top to close any open paths before we
      // start the next.
      curSeg=pi.isDone()? PathIterator.SEG_CLOSE: pi.currentSegment(segment);
      if (pi.isDone() || (open && curSeg == PathIterator.SEG_MOVETO)) {
	// first close the path
	curSeg = PathIterator.SEG_CLOSE;
      } else
	// only advance if we're not doing a special 'path closing' segment.
	pi.next();
      // the shape is still open if it hasn't been closed.
      open = !(curSeg == PathIterator.SEG_CLOSE ||
	       curSeg == PathIterator.SEG_MOVETO);

      // okay, now handle the path (as if all paths were closed)
      switch(curSeg) {
      case PathIterator.SEG_MOVETO:
	start.setLocation(segment[0], segment[1]);
	last.setLocation(segment[0], segment[1]);
	continue; // keep going.
      case PathIterator.SEG_CLOSE:
	segment[0] = start.getX();
	segment[1] = start.getY();
      case PathIterator.SEG_LINETO:
	{
	  /* check if endpoints straddle the X axis; if so a +X ray could
	   * intersect this edge. */
	  boolean yflag0 = last.getY() >= y;
	  boolean yflag1 = segment[1] >= y;
	  if (yflag0 != yflag1) {
	    /* check if endpoints are on same side of the Y axis (i.e. X's are
	     * the same.  If so, it's easy to test if edge hits or misses. */
	    boolean xflag0 = last.getX() >= x;
	    boolean xflag1 = segment[0] >= x;
	    if (xflag0 == xflag1) {
	      /* if edge's X values both right of the point, must hit */
	      if (xflag0) {
		crossings += yflag0 ? -1 : 1;
		inside_flag = !inside_flag;
	      }
	    } else {
	      /* compute intersection of polygon segment with +X ray, note
	       * if >= point's X; if so, the ray hits it. */
	      if ( (segment[0] - (segment[1]-y)*
		    (last.getX()-segment[0])/(last.getY()-segment[1])) >= x) {
		crossings += yflag0 ? -1 : 1;
		inside_flag = !inside_flag;
	      }
	    }
	  }
	}
	last.setLocation(segment[0], segment[1]);
	break;
      case PathIterator.SEG_QUADTO:
	// raise quad curve to a cubic and use SEG_CUBICTO
	double[] param = new double[8];
	param[0] = last.getX(); param[1] = last.getY();
	System.arraycopy(segment, 0, param, 2, 4);
	CubicCurve2D.raiseDegree(param, 2, param, 3);
	System.arraycopy(param, 2, segment, 0, 6);
	// fall through.
      case PathIterator.SEG_CUBICTO:
	{
	  // solve cubic for x-axis intercepts.
	  double P0 = last.getY();
	  double P1 = segment[1];
	  double P2 = segment[3];
	  double P3 = segment[5];
	  // y(t) = (-P0+3P1-3P2+P3)t^3 + (3P0-6P1+3P2)t^2 + (-3P0+3P1)t + P0
	  // find roots of y(t) - y
	  double[] roots = new double[3];
	  double t3 = -P0 + 3*P1 - 3*P2 + P3;
	  double t2 = 3*(P0-2*P1+P2);
	  double t1 = 3*(-P0+P1);
	  double t0 = P0-y;
	  int num = CubicCurve2D.solveCubic(new double[] { t0, t1, t2, t3 },
					    roots);
	  for (int i=0; i<num; i++) {
	    if (roots[i] < 0 || roots[i] > 1) continue; // not on curve segment
	    // compute x coord and make sure it is >= point's x.
	    double onemt = (1-roots[i]);
	    double cx = last.getX()*onemt*onemt*onemt +
	      roots[i]*(3*segment[0]*onemt*onemt +
			roots[i]*(3*segment[2]*onemt +
				  roots[i]*segment[4]));
	    if (cx < x) continue; // not on +X ray.
	    // compute y derivative at root.
	    //  dy(t) = 3*(-P0+3P1-3P2+P3)t^2 + 2*(3P0-6P1+3P2)t + (-3P0+3P1)
	    double derivative = (3*t3*roots[i] + 2*t2)*roots[i] + t1;
	    if (derivative != 0) {
	      crossings += (derivative > 0) ? 1 : -1;
	      inside_flag = !inside_flag;
	    }
	  }
	}
	last.setLocation(segment[4], segment[5]);
	break;
      default:
	assert false;
      }
    }
    if (rule==WIND_NON_ZERO)
      inside_flag = (crossings != 0);
    return inside_flag;
  }
  public boolean contains(Point2D p)
  {
    return contains(p.getX(), p.getY());
  }
  public boolean contains(double x, double y, double w, double h)
  {
    if (!getBounds2D().contains(x,y,w,h)) return false; // fast path.
    if (rule==WIND_NON_ZERO) {
      // yuck.  our rule below for WIND_EVEN_ODD doesn't work for
      // WIND_NON_ZERO because there might be entirely-interior
      // edges which intersect our box.  So punt this to Area.
      Area a = new Area(new Rectangle2D.Double(x,y,w,h));
      a.subtract(new Area(this));
      return a.isEmpty();
    }
    // Find if any edge of box intersects any edge of polygon.  If so,
    // then return false.  Otherwise, rectangle is either entirely
    // *outside* or entirely *inside* polygon.  Test an arbitrary vertex
    // of the rectangle using contains().  Return true iff the polygon
    // contains that point.
    assert rule==WIND_EVEN_ODD;
    if (intersectsLine(x, y, x+w, y) || intersectsLine(x+w, y, x+w, y+h) ||
	intersectsLine(x, y+h, x+w, y+h) || intersectsLine(x, y, x, y+h))
      return false;
    return contains(x, y);
  }
  public boolean contains(Rectangle2D r)
  {
    return contains(r.getX(), r.getY(), r.getWidth(), r.getHeight());
  }

  public boolean intersects(double x, double y, double w, double h)
  {
    if (!getBounds2D().intersects(x,y,w,h)) return false; // fast path.
    // Find if any edge of box intersects any edge of
    // polygon.  If so, then return true.  Otherwise, rectangle
    // is either entirely *outside* or entirely *inside* the polygon.
    // Test an arbitrary vertex of the rectangle using contains().
    // Return true if the polygon contains that point.
    // Finally: test an arbitrary vertex of the polygon; return true
    // iff the rectangle contains that point.
    if (intersectsLine(x, y, x+w, y) || intersectsLine(x+w, y, x+w, y+h) ||
	intersectsLine(x, y+h, x+w, y+h) || intersectsLine(x, y, x, y+h))
      return true;
    if (contains(x,y)) return true;
    Point2D pt = getCurrentPoint();
    if (pt==null) return false; // no area in this path!
    // this could be more efficient.
    return new Rectangle2D.Double(x,y,w,h).contains(pt.getX(), pt.getY());
  }
  public boolean intersects(Rectangle2D r)
  {
    return intersects(r.getX(), r.getY(), r.getWidth(), r.getHeight());
  }
  
  /**
   * Return true if some edge in this GeneralPath intersects the given
   * line.  Otherwise, return false.
   * <p>
   * GNU extension. */
  public boolean intersectsLine(Line2D l) {
    return intersectsLine(l.getX1(), l.getY1(), l.getX2(), l.getY2());
  }
  /**
   * Return true if some edge in this GeneralPath intersects the line
   * from (x1, y1) to (x2, y2).  Otherwise, return false.
   * <p>
   * GNU extension. */
  public boolean intersectsLine(double x1, double y1, double x2, double y2) {
    Line2D testline = new Line2D.Double(x1, y1, x2, y2);

    Point2D last = new Point2D.Double(), start = new Point2D.Double();
    double[] segment = new double[6];
    for (PathIterator pi=getPathIterator(null); !pi.isDone(); pi.next()) {
      switch(pi.currentSegment(segment)) {
      case PathIterator.SEG_MOVETO:
	start.setLocation(segment[0], segment[1]);
	last.setLocation(segment[0], segment[1]);
	continue; // keep going.
      case PathIterator.SEG_CLOSE:
	segment[0] = start.getX();
	segment[1] = start.getY();
      case PathIterator.SEG_LINETO:
	if (new Line2D.Double(last.getX(), last.getY(), segment[0], segment[1])
	    .intersectsLine(testline))
	  return true;
	last.setLocation(segment[0], segment[1]);
	break;
      case PathIterator.SEG_QUADTO:
	if (new QuadCurve2D.Double(last.getX(), last.getY(),
				   segment[0], segment[1],
				   segment[2], segment[3])
	    .intersectsLine(testline))
	  return true;
	last.setLocation(segment[2], segment[3]);
	break;
      case PathIterator.SEG_CUBICTO:
	if (new CubicCurve2D.Double(last.getX(), last.getY(),
				    segment[0], segment[1],
				    segment[2], segment[3],
				    segment[4], segment[5])
	    .intersectsLine(testline))
	  return true;
	last.setLocation(segment[4], segment[5]);
	break;
      default:
	assert false;
      }
    }
    return false;
  }

  public PathIterator getPathIterator(final AffineTransform at)
  {
    return new PathIterator()
    {
      int current = 0;

      public int getWindingRule()
      {
        return rule;
      }

      public boolean isDone()
      {
        return current >= index;
      }

      public void next()
      {
	switch(types[current>>1]) {
	case PathIterator.SEG_CUBICTO:
	  current+=2;
	  // fall through.
	case PathIterator.SEG_QUADTO:
	  current+=2;
	  // fall through.
	case PathIterator.SEG_LINETO:
	case PathIterator.SEG_MOVETO:
	case PathIterator.SEG_CLOSE: // close has points too!
	  current+=2;
	  break;
	default:
	  assert false : "what segment type is this?";
	}
      }

      public int currentSegment(float[] coords)
      {
        if (current >= index)
          return SEG_CLOSE;
        int result = types[current >> 1];
        int i = 0, j = current;
	switch (result) {
	case PathIterator.SEG_CUBICTO:
            coords[i++] = (float) points[j++];
            coords[i++] = (float) points[j++];
	    // fall through.
	case PathIterator.SEG_QUADTO:
            coords[i++] = (float) points[j++];
            coords[i++] = (float) points[j++];
	    // fall through.
	case PathIterator.SEG_LINETO:
	case PathIterator.SEG_MOVETO:
	case PathIterator.SEG_CLOSE: // close has points too!
            coords[i++] = (float) points[j++];
            coords[i++] = (float) points[j++];
	    break;
	default:
	  assert false : "what segment type is this? "+result;
	}
	if (at != null)
	  at.transform(coords, 0, coords, 0, i/2);
        return result;
      }

      public int currentSegment(double[] coords)
      {
        if (current >= index)
          return SEG_CLOSE;
        int result = types[current >> 1];
        int i = 0, j = current;
	switch (result) {
	case PathIterator.SEG_CUBICTO:
            coords[i++] = points[j++];
            coords[i++] = points[j++];
	    // fall through.
	case PathIterator.SEG_QUADTO:
            coords[i++] = points[j++];
            coords[i++] = points[j++];
	    // fall through.
	case PathIterator.SEG_LINETO:
	case PathIterator.SEG_MOVETO:
	case PathIterator.SEG_CLOSE: // close has points too!
            coords[i++] = points[j++];
            coords[i++] = points[j++];
	    break;
	default:
	  assert false : "what segment type is this? "+result;
	}
	if (at != null)
	  at.transform(coords, 0, coords, 0, i/2);
        return result;
      }
    };
  }
  public PathIterator getPathIterator(AffineTransform at, double flatness)
  {
    return new FlatteningPathIterator(getPathIterator(at), flatness);
  }

  /**
   * Create a new shape of the same run-time type with the same contents as
   * this one.
   *
   * @return the clone
   */
  public Object clone()
  {
    // This class is final; no need to use super.clone().
    return new GeneralPath(this);
  }

  private void ensureSize(int size)
  {
    if (subpath < 0)
      throw new IllegalPathStateException("need initial moveto");
    if (size <= points.length)
      return;
    // the new size should be double the old, but at least 'size'.
    int newsize = Math.max(size, points.length*2);
    byte[] b = new byte[newsize>>1];
    System.arraycopy(types, 0, b, 0, index >> 1);
    types = b;
    double[] f = new double[newsize];
    System.arraycopy(points, 0, f, 0, index);
    points = f;
  }
} // class GeneralPath
