package net.cscott.pcbmill;

import java.awt.Shape;
import java.awt.geom.*;
import java.io.*;
import java.util.*;

import gnu.java.awt.geom.CubicCurve2D;

/** This class creates DXF output for (the outline of) an input Shape. */
// XXX cubic->arc conversion not yet implemented.
public abstract class DXFOutput {
    /** Write the given shape as a DXF file to the given Writer.  If
     *  'useArcs' is true, then curves will be approximated with
     *  elliptical arc segments.  Otherwise they will be approximated
     *  with straight lines. */
    public static void writeDXF(Writer w, Shape s, boolean useArcs) {
	useArcs=false; // not yet implemented.
	PrintWriter pw = new PrintWriter(w);
	initDXF(pw, s.getBounds2D());
	outlineShape(pw, s, useArcs);
	closeDXF(pw);
	pw.flush();
    }

    private static void initDXF(PrintWriter w, Rectangle2D bounds) {
	// HEADER section
	doSection(w, "HEADER");
	//  "autocad drawing datavase version number"
	doHeaderVariable(w, "$ACADVER", 1, "AC1003"); // autocad 2.6
	//   drawing extents lower-left corner
	doGroup(w, 9, "$EXTMIN");
	doGroup(w, 10, bounds.getMinX());
	doGroup(w, 20, bounds.getMaxY()); // this is what OMAX does.
	//   drawing extents upper-right corner
	doGroup(w, 9, "$EXTMAX");
	doGroup(w, 10, bounds.getMaxX());
	doGroup(w, 20, bounds.getMinY());
	//   "insertion base set by BASE command (in WCS)"
	doGroup(w, 9, "$INSBASE");
	doGroup(w, 10, (bounds.getMinX()+bounds.getMaxX())/2);
	doGroup(w, 20, (bounds.getMinY()+bounds.getMaxY())/2);
	//   comment.
	//doGroup(w, 999, "Created with "+Version.PACKAGE_STRING+" "+new Date());
	doGroup(w, 0, "ENDSEC");

	// CLASSES section
	doSection(w, "CLASSES");
	doGroup(w, 0, "ENDSEC");

	// TABLES section
	doSection(w, "TABLES");
	doGroup(w, 0, "TABLE"); // start a table
	doGroup(w, 2, "LAYER"); // the type of table
	doGroup(w, 70, 1);    // number of items in the table
	doGroup(w, 0, "LAYER"); // start of table entry.
	doGroup(w, 2, "0"); // table name??
	doGroup(w, 70, 0); // number of items in the layer definition?
	doGroup(w, 62, 7); // layer color??
	doGroup(w, 6, "CONTINUOUS"); // line type for the layer??
	doGroup(w, 0, "ENDTAB"); // end of table
	doGroup(w, 0, "ENDSEC");

	// BLOCKS section
	doSection(w, "BLOCKS");
	doGroup(w, 0, "ENDSEC");

	// ENTITIES section
	doSection(w, "ENTITIES");
	// leave the entities section open so we can put our shapes in here.
    }
    private static void closeDXF(PrintWriter w) {
	// close up the ENTITIES section.
	doGroup(w, 0, "ENDSEC");

	// we're going to skip the OBJECTS section.

	doGroup(w, 0, "EOF"); // done!
    }
    private static void outlineShape(PrintWriter w, Shape s, boolean useArcs) {
	PathIterator pi= useArcs ? s.getPathIterator(null)
	    : s.getPathIterator(null, 0.001);
	double[] coords = new double[6];
	double startx=0, starty=0, lastx=0, lasty=0;
	for (; !pi.isDone(); pi.next()) {
	    switch(pi.currentSegment(coords)) {
	    case PathIterator.SEG_MOVETO:
		startx = coords[0]; starty = coords[1];
		lastx = coords[0]; lasty = coords[1];
		break;
	    case PathIterator.SEG_CLOSE:
		coords[0] = startx; coords[1] = starty;
		// fall through.
	    case PathIterator.SEG_LINETO:
		doLine(w, lastx, lasty, coords[0], coords[1]);
		lastx = coords[0]; lasty = coords[1];
		break;
	    case PathIterator.SEG_QUADTO:
		// raise quad curve to a cubic and use SEG_CUBICTO
		double[] param = new double[8];
		param[0] = lastx; param[1]=lasty;
		System.arraycopy(coords, 0, param, 2, 4);
		CubicCurve2D.raiseDegree(param, 2, param, 3);
		System.arraycopy(param, 2, coords, 0, 6);
		// fall through
	    case PathIterator.SEG_CUBICTO:
		assert useArcs;
		doQuad(w, lastx, lasty, coords[0], coords[1],
		       coords[2], coords[3], coords[4], coords[5]);
		lastx = coords[4]; lasty = coords[5];
		break;
	    default:
		assert false;
	    }
	}
	// done!
	w.flush();
    }
    private static void doLine(PrintWriter w,
			       double x1, double y1, double x2, double y2) {
	doGroup(w, 0, "LINE"); // entity type.
	doGroup(w, 8, "0"); // layer name
	doGroup(w, 6, "CONTINUOUS"); // linetype name
	doGroup(w, 62, "0"); // color number
	// group code 10 = start point x value
	doGroup(w, 10, x1);
	// group code 20 = start point y value
	doGroup(w, 20, y1);
	// group code 30 = start point z value
	doGroup(w, 30, 0.0);
	// group code 11 = end point x value
	doGroup(w, 11, x2);
	// group code 21 = end point y value
	doGroup(w, 21, y2);
	// group code 31 = end point z value
	doGroup(w, 31, 0.0);
    }
    private static void doQuad(PrintWriter w,
			       double x1, double y1, double x2, double y2,
			       double x3, double y3, double x4, double y4) {
	// XXX convert to an arc and then call doArc()
	throw new Error("unimplemented");
    }
    private static void doArc(PrintWriter w,
			      double centerX, double centerY, double radius,
			      double startAngle, double endAngle) {
	doGroup(w, 0, "ARC"); // entity type.
	doGroup(w, 8, "0"); // layer name
	doGroup(w, 6, "CONTINUOUS"); // linetype name
	doGroup(w, 62, "0"); // color number
	// group code 10 = center point x value
	doGroup(w, 10, centerX);
	// group code 20 = center point y value
	doGroup(w, 20, centerY);
	// group code 30 = center point z value
	doGroup(w, 30, 0.0);
	// group code 40 = radius
	doGroup(w, 40, radius);
	// group code 50 = start angle in degrees
	doGroup(w, 50, Math.toDegrees(startAngle));
	// group code 51 = end angle in degrees
	doGroup(w, 51, Math.toDegrees(endAngle));
    }
    /////////////////////
    private static void doGroup(PrintWriter w, int group, long value) {
	doGroup(w, group, Long.toString(value));
    }
    private static void doGroup(PrintWriter w, int group, double value) {
	doGroup(w, group, Double.toString(value));
    }
    private static void doGroup(PrintWriter w, int group, String value) {
	w.println(" "+group);
	w.println(value);
    }
    //
    private static void doSection(PrintWriter w, String section_name) {
	doGroup(w, 0, "SECTION");
	doGroup(w, 2, section_name);
    }
    private static void doHeaderVariable(PrintWriter w, String variable_name,
			  int group, String value) {
	doGroup(w, 9, variable_name);
	doGroup(w, group, value);
    }
}
