/* CubicCurve2D.java -- represents a parameterized cubic curve in 2-D space
   Copyright (C) 2002 Free Software Foundation

This file is part of GNU Classpath.

GNU Classpath is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU Classpath is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under
terms of your choice, provided that you also meet, for each linked
independent module, the terms and conditions of the license of that
module.  An independent module is a module which is not derived from
or based on this library.  If you modify this library, you may extend
this exception to your version of the library, but you are not
obligated to do so.  If you do not wish to do so, delete this
exception statement from your version. */


package gnu.java.awt.geom;
// --- this stuff can be removed once the package is changed back
import java.awt.geom.*;
import gnu.java.awt.geom.Line2D;
import gnu.java.awt.geom.QuadCurve2D;
// --- end stuff to be removed.

import java.awt.Rectangle;
import java.awt.Shape;
import java.util.NoSuchElementException;

/**
 * XXX Document.
 */
public abstract class CubicCurve2D implements Shape, Cloneable
{
  protected CubicCurve2D()
  {
  }

  public abstract double getX1();
  public abstract double getY1();
  public abstract Point2D getP1();
  public abstract double getCtrlX1();
  public abstract double getCtrlY1();
  public abstract Point2D getCtrlP1();
  public abstract double getCtrlX2();
  public abstract double getCtrlY2();
  public abstract Point2D getCtrlP2();
  public abstract double getX2();
  public abstract double getY2();
  public abstract Point2D getP2();

  public abstract void setCurve(double x1, double y1, double cx1, double cy1,
                                double cx2, double cy2, double x2, double y2);
  public void setCurve(double[] coords, int offset)
  {
    setCurve(coords[offset++], coords[offset++],
             coords[offset++], coords[offset++],
             coords[offset++], coords[offset++],
             coords[offset++], coords[offset++]);
  }
  public void setCurve(Point2D p1, Point2D c1, Point2D c2, Point2D p2)
  {
    setCurve(p1.getX(), p1.getY(), c1.getX(), c1.getY(),
             c2.getX(), c2.getY(), p2.getX(), p2.getY());
  }
  public void setCurve(Point2D[] pts, int offset)
  {
    setCurve(pts[offset].getX(), pts[offset++].getY(),
             pts[offset].getX(), pts[offset++].getY(),
             pts[offset].getX(), pts[offset++].getY(),
             pts[offset].getX(), pts[offset++].getY());
  }
  public void setCurve(CubicCurve2D c)
  {
    setCurve(c.getX1(), c.getY1(), c.getCtrlX1(), c.getCtrlY1(),
             c.getCtrlX2(), c.getCtrlY2(), c.getX2(), c.getY2());
  }
  // flatness is max distance from line <x1,y1>-><x2,y2> of a control point.
  public static double getFlatnessSq(double x1, double y1, double cx1,
                                     double cy1, double cx2, double cy2,
                                     double x2, double y2)
  {
    // use point-to-line distance function in Line2D.
    double d1 = Line2D.ptLineDistSq(x1,y1,x2,y2,cx1,cy1);
    double d2 = Line2D.ptLineDistSq(x1,y1,x2,y2,cx2,cy2);
    return Math.max(d1, d2);
  }
  public static double getFlatness(double x1, double y1, double cx1,
                                   double cy1, double cx2, double cy2,
                                   double x2, double y2)
  {
    return Math.sqrt(getFlatnessSq(x1, y1, cx1, cy1, cx2, cy2, x2, y2));
  }
  public static double getFlatnessSq(double[] coords, int offset)
  {
    return getFlatnessSq(coords[offset++], coords[offset++],
                         coords[offset++], coords[offset++],
                         coords[offset++], coords[offset++],
                         coords[offset++], coords[offset++]);
  }
  public static double getFlatness(double[] coords, int offset)
  {
    return Math.sqrt(getFlatnessSq(coords[offset++], coords[offset++],
                                   coords[offset++], coords[offset++],
                                   coords[offset++], coords[offset++],
                                   coords[offset++], coords[offset++]));
  }
  public double getFlatnessSq()
  {
    return getFlatnessSq(getX1(), getY1(), getCtrlX1(), getCtrlY1(),
                         getCtrlX2(), getCtrlY2(), getX2(), getY2());
  }
  public double getFlatness()
  {
    return Math.sqrt(getFlatnessSq(getX1(), getY1(), getCtrlX1(),
                                   getCtrlY1(), getCtrlX2(), getCtrlY2(),
                                   getX2(), getY2()));
  }

  public void subdivide(CubicCurve2D l, CubicCurve2D r)
  {
    subdivide(l,r,.5/* divide at midpoint */);
  }
  /** GNU extension. */
  public void subdivide(CubicCurve2D l, CubicCurve2D r, double t)
  {
    if (l == null)
      l = new CubicCurve2D.Double();
    if (r == null)
      r = new CubicCurve2D.Double();
    // Use empty slots at end to share single array.
    double[] d = new double[] { getX1(), getY1(), getCtrlX1(), getCtrlY1(),
                                getCtrlX2(), getCtrlY2(), getX2(), getY2(),
                                0, 0, 0, 0, 0, 0, 0, 0 };
    subdivide(d, 0, d, 0, d, 8, t);
    l.setCurve(d, 0);
    r.setCurve(d, 8);
  }
  public static void subdivide(CubicCurve2D src,
                               CubicCurve2D l, CubicCurve2D r)
  {
    src.subdivide(l, r);
  }
  /** GNU extension. */
  public static void subdivide(CubicCurve2D src,
                               CubicCurve2D l, CubicCurve2D r, double t)
  {
    src.subdivide(l, r, t);
  }
  public static void subdivide(double[] src, int srcOff,
                               double[] left, int leftOff,
                               double[] right, int rightOff)
  {
    /* divide at midpoint (t=0.5) */
    subdivide(src, srcOff, left, leftOff, right, rightOff, .5);
  }
  // subdivision by de Casteljau's algorithm; see
  // Hartmut Prautzsch, "A round trip to B-splines via de Casteljau"
  // ACM Transactions on Graphics (TOG), vol 8 num 3 1989 243--254
  //    http://doi.acm.org/10.1145/77055.77061
  /** GNU extension. */
  public static void subdivide(double[] src, int srcOff,
                               double[] left, int leftOff,
                               double[] right, int rightOff, double t)
  {
    // do subdivision twice, one on the x and once on the y values.
    subdivide(src, srcOff,   left, leftOff,   right, rightOff,   t, 3, 2);
    subdivide(src, srcOff+1, left, leftOff+1, right, rightOff+1, t, 3, 2);
  }
  // this is general degree-n bezier subdivision at parameter t in (0,1)
  // step tells how many parallel bezier equations are in the
  // src/left/right arrays.  for standard x/y point format, step=2.
  static void subdivide(double[] src, int srcOff,
			double[] left, int leftOff,
			double[] right, int rightOff,
			double t, int degree, int step)
  {
    final int n=degree+1; // degree=3, n=4 for bezier curves.
    // C b = (1-t)[b_0,...,b_n-1] + t[b_1,...,b_n]
    // apply this n times to get C^n b
    // left = first points of C^i b  for i=0,1,...,n
    // right = last points of C^i b  for i=0,1,...,n

    double[] cb = new double[n];
    for (int i=0; i<cb.length; i++)
      cb[i] = src[srcOff + (i*step)];

    for (int i=0; i<n; i++) {
      // take first point in cb and put it in left.
      left[leftOff+step*i] = cb[0];
      // take last point in cb and put it in right.
      right[rightOff+step*(n-1-i)] = cb[n-1-i];
      // now compute next level of de Casteljau triangle.
      for (int j=0; j<n-1-i; j++)
	cb[j] = (1-t)*cb[j] + t*cb[j+1];
    }
    // done!
  }

  // Offset a bezier curve.  Surprisingly, this can't be done exactly.
  // Our reference here is:
  //  "Comparing Offset Curve Approximation Methods"
  //  Gershon Elber, In-Kwon Lee, and Myung-Soo Kim
  //  IEEE Computer Graphics and Applications, May-June 1997.  p 62-71
  //     http://3map.snu.ac.kr/mskim/ftp/comparing.pdf
  /** GNU extension. */
  public Shape offset(double dist, Tolerance tolerance) {
    double max_error = Math.max(tolerance.absolute_precision,
				tolerance.relative_precision*dist/100);
    GeneralPath gp = new GeneralPath();
    _offset_(gp, dist, max_error, true/* first in the path*/);
    return gp;
  }
  // append the offset path to 'gp'
  private void _offset_(GeneralPath gp, double dist, double max_error,
			boolean first_in_path) {
    // we use Tiller and Hanson's method, which offsets each *edge* of
    // the control polygon.  This is exact for lines.

    // control polygon = P0->P1->P2->P3->close
    // P'0 = endpoint of offset(P0->P1)
    // P'1 = intersection of offset(P0->P1) and offset(P1->P2)
    // P'2 = intersection of offset(P1->P2) and offset(P2->P3)
    // P'3 = endpoint of offset(P2->P3)
    
    Line2D P0P1 = new Line2D.Double
      (getX1(), getY1(), getCtrlX1(), getCtrlY1()).offset(dist);
    Line2D P1P2 = new Line2D.Double
      (getCtrlX1(), getCtrlY1(), getCtrlX2(), getCtrlY2()).offset(dist);
    Line2D P2P3 = new Line2D.Double
      (getCtrlX2(), getCtrlY2(), getX2(), getY2()).offset(dist);

    Point2D nP0 = P0P1.getP1();
    Point2D nP1 = P0P1.intersection(P1P2);
    Point2D nP2 = P1P2.intersection(P2P3);
    Point2D nP3 = P2P3.getP2();

    // deal with degenerate cases.
    if (nP1==null) nP1 = P1P2.getP1();
    if (nP2==null) nP2 = P2P3.getP1();

    // see if curve is "close enough"; if not, then subdivide
    // and recompute.
    if (error_within_bounds(new double[] { getX1(), getY1(),
					   getCtrlX1(), getCtrlY1(),
					   getCtrlX2(), getCtrlY2(),
					   getX2(), getY2() }, 3,
			    new double[] { nP0.getX(), nP0.getY(),
					   nP1.getX(), nP1.getY(),
					   nP2.getX(), nP2.getY(),
					   nP3.getX(), nP3.getY() }, 3,
			    dist, max_error)) {
      // error is small enough, return this curve.
      if (first_in_path)
	gp.moveTo(nP0.getX(), nP0.getY());
      gp.curveTo(nP1.getX(), nP1.getY(),
		 nP2.getX(), nP2.getY(),
		 nP3.getX(), nP3.getY());
    } else {
      // error is too large.  subdivide on max t and try again.
      CubicCurve2D left = new CubicCurve2D.Double();
      CubicCurve2D right = new CubicCurve2D.Double();
      this.subdivide(left, right, 0.5);
      left._offset_(gp, dist, max_error, first_in_path);
      right._offset_(gp, dist, max_error, false);
    }
  }

  private static boolean error_within_bounds(double[] bezier1, int degree1,
					     double[] bezier2, int degree2,
					     double desired_offset,
					     double max_error) {
    // error function:
    //  e(t) = || C1(t) - C2(t) ||^2 - desired_offset

    // use raiseDegree to make them both the same degree.
    int degree = Math.max(degree1, degree2);
    double[] bz1 = new double[2*(degree+1)];
    double[] bz2 = new double[2*(degree+1)];
    raiseDegree(bezier1, degree1, bz1, degree);
    raiseDegree(bezier2, degree2, bz2, degree);
    // okay, now we just subtract the beziers.  Equivalent to subtracting
    // the control points.
    for (int i=0; i<2*(degree+1); i++)
      bz1[i] -= bz2[i];
    bz2=null; // just to clarify that we don't need it anymore.
    //  Now we want to take the dot product of bz1 with itself to
    //  get || C1(t) - C2(t) ||^2
    //  Following the "Comparing Offset Curve Approximation Methods" paper
    //  (referenced above) we have:
    //   C1(t) = SUM P1_i B_i,m (t) for i=0 to m
    //   C2(t) = SUM P2_j B_j,n (t) for j=0 to n
    //   C1(t)C2(t) = SUM Q_k B_k,(m+n)(t) from k=0 to m+n
    //   where
    //     Q_k = SUM P1_i P2_(k-i) (m choose i)(n choose k-i) / (m+n choose k)
    //           for i=0 to k

    // note that the result of multiplying two degree-3 beziers is a degree-6
    // bezier.

    // multiply x components w/ x components, and then add multiplication of
    // y components w/ y components.
    //  so on first pass   P1_i = bz1[2*i] and P2_j = bz1[2*j]
    //  and on second pass P1_i = bz1[2*i+1] and P2_j = bz1[2*j+1]
    // always m=n=degree.
    int m=degree, n=degree;
    double[] bzprod = new double[m+n+1];
    for (int xy=0; xy<=1; xy++) {
      for (int k=0; k<=m+n; k++) {
	for (int i=Math.max(0,k-n); i<=Math.min(k,m); i++) {
	  bzprod[k]+= bz1[2*i+xy] * bz1[2*(k-i)+xy] *
	    choose(m,i) * choose(n,k-i) / choose(m+n,k);
	}
      }
    }

    // okay, now we have a degree '2*degree' polynomial for the squared
    // distance between bezier1 and bezier2.  Let's subtract our
    // desired offset, squared.  To subtract a constant from a bezier, subtract
    // it from each control point.
    for (int i=0; i<bzprod.length; i++)
      bzprod[i] -= (desired_offset*desired_offset);

    // see if maximum is within bounds.
    return !exceedsBounds(bzprod, bzprod.length-1,
			  /* distance squared */ max_error * max_error, 0);
  }
  // helper functions for error_within_bounds
  /** Returns "a choose b" = a!/((a-b)!b!). */
  private static double choose(int a, int b) {
    return fact(a)/(fact(a-b)*fact(b));
  }
  /** Returns "a factorial" = 1*2*...*a. */
  private static double fact(int a) {
    if (a==0) return 1;
    double r = a;
    for (int i=2; i<a; i++)
      r*=i;
    return r;
  }

  /** Find if the absolute value of the given bezier curve exceeds
   *  stop_value at any point.
   *  If it does, return true.  Else return false.
   *  Subdivide and try again if not sure.
   */
  private static boolean exceedsBounds(double[] bezier, int degree,
				       double stop_value,
				       int recursion_depth)
  {
    // check end points, since this bezier will pass through them.
    if (bezier[0] > stop_value ||
	bezier[degree] > stop_value)
      return true; // this bezier exceeds its bounds at an endpoint.
    // look at absolute value of the control polygon.
    // use the fact that control polygon encloses bezier.
    double min, max;
    min = max = Math.abs(bezier[0]);
    for (int i=1; i<=degree; i++) {
      double v = Math.abs(bezier[i]);
      if (v < min) min = v;
      if (v > max) max = v;
    }
    // min/max are min/max possible *error* for this bezier piece.
    if (max < stop_value) return false; // no points larger than stop_value
    // note that we can't conclude anything about min>stop_value unless
    // we know there aren't any zero crossings.
    if (min > stop_value && bezierCrossings(bezier, degree)==0)
      return true; // all points in this piece are large.
    // play it safe, in case we have a straight line exactly on stop_value.
    if (recursion_depth > MAX_RECURSION_DEPTH)
      return true; // be safe.

    // okay, none of the simple tests worked.  Let's subdivide this
    // bezier and try again on the left and right halves.

    double[] left_bezier = new double[degree+1];
    double[] right_bezier = new double[degree+1];
    subdivide(bezier, 0, left_bezier, 0, right_bezier, 0, 0.5, degree, 1);

    // do "larger" half first for speed (make larger half the "left" half)
    if (Math.max(Math.abs(left_bezier[0]),Math.abs(left_bezier[degree])) < 
	Math.max(Math.abs(right_bezier[0]),Math.abs(right_bezier[degree]))) {
      double[] tmp = left_bezier; left_bezier=right_bezier; right_bezier=tmp;
    }

    // test left half.
    if (exceedsBounds(left_bezier, degree, stop_value, recursion_depth+1))
      return true;
    // test right half
    if (exceedsBounds(right_bezier, degree, stop_value, recursion_depth+1))
      return true;
    // well, I guess we're inside our bounds then!
    return false;
  }
  // protect ourselves against really, really flat beziers that just *refuse*
  // to stay on one side of stop_value or another.
  private static final int MAX_RECURSION_DEPTH=7;
  /** Return the number of times a Bezier control polygon crosses the
   *  0-axis.  This number is >= the number of roots. */
  private static int bezierCrossings(double[] bezier, int degree) {
    int n_crossings = 0;
    int sign, old_sign;
    sign = old_sign = (bezier[0] == 0) ? 0 : (bezier[0] < 0) ? -1 : 1;
    for (int i=1; i<=degree; i++) {
      sign = (bezier[i] == 0) ? 0 : (bezier[i] < 0) ? -1 : 1;
      if (sign != old_sign) n_crossings++;
      old_sign = sign;
    }
    return n_crossings;
  }

  // degree raising of a bezier polynomial expressed as a set of control
  // points.  This algorithm is from 
  //    http://www.cis.upenn.edu/~cis510/cis510sl6.ps
  // If curve is specified by m+1 control points b_0,...,b_m, then
  // the degree m+1 polynomial is specified by the m+2 control points
  // c_0,...,c_m+1 where
  //  c_i = (i/(m+1)) b_(i-1) + ((m+1-i)/(m+1)) b_i   for 1<=i<=m
  // and c_0=b_0 and c_(m+1)=b_m
  // this can be done iteratively.
  // NOTE THAT in must contain (in_degree+1) pairs of points!
  /** GNU extension. */
  public static void raiseDegree(double[] in, int in_degree,
				 double[] out, int out_degree) {
    // note that we could have in==out.
    assert in_degree <= out_degree;
    assert in.length >= 2*(in_degree+1);
    assert out.length >= 2*(out_degree+1);
    System.arraycopy(in, 0, out, 0, 2*(in_degree+1));
    in=out; // from now on we'll just use in and out as equivalent.
    for (int m=in_degree; m<out_degree; m++) {
      // raise degree one step at a time.
      // c_m+1 = b_m:
      out[2*(m+1)] = in[2*m]; // x coordinate
      out[2*(m+1)+1] = in[2*m+1]; // y coordinate
      // c_i = (i/(m+1)) b_(i-1) + ((m+1-i)/(m+1)) b_i   for 1<=i<=m
      for (int i=m; i>=1; i--) {
	out[2*i] = (i*in[2*(i-1)] + (m+1-i)*in[2*i]) / (m+1); // x
	out[2*i+1] = (i*in[2*(i-1)+1] + (m+1-i)*in[2*i+1]) / (m+1); // y
      }
      // now c_0 = b_0.  Well, it already is, so we don't need to change it.
    }
    // done!
  }

  public static int solveCubic(double[] eqn)
  {
    return solveCubic(eqn, eqn);
  }
  // algorithm from "Graphics Gems" volume 1, p 404,738.
  public static int solveCubic(double[] eqn, double[] res)
  {
    int num=0; // number of solutions found.

    // eqn = {c, b, a, d} for dx^3 + ax^2 + bx + c = 0
    if (eqn[3] == 0)
      return QuadCurve2D.solveQuadratic(eqn, res);

    /* normal form: x^3 + Ax^2 + Bx + C = 0 */
    double A = eqn[2] / eqn[3];
    double B = eqn[1] / eqn[3];
    double C = eqn[0] / eqn[3];

    // substitute x = y - A/3 to eliminate quadric term:
    //     x^3 + px + q = 0
    double sq_A = A*A;
    double p = (1.0/3) * (-1.0/3 * sq_A + B);
    double q = (1.0/2) * (2.0/27 * A * sq_A - 1.0/3 * A * B + C);
    
    // Use Cardano's formula
    double cb_p = p * p * p;
    double D = q*q + cb_p;

    if (isZero(D)) {
      if (isZero(q)) { /* one triple solution */
	res[0] = 0;
	num = 1;
      } else { /* one single and one double solution */
	double u = cbrt(-q);
	res[0] = 2*u;
	res[1] = -u;
	num = 2;
      }
    } else if (D<0) { /* Casus irreducibilis: three real solutions */
      double phi = 1.0/3 * Math.acos(-q/Math.sqrt(-cb_p));
      double t = 2 * Math.sqrt(-p);

      res[0] =  t * Math.cos(phi);
      res[1] = -t * Math.cos(phi + Math.PI/3);
      res[2] = -t * Math.cos(phi - Math.PI/3);
      num = 3;
    } else { /* one real solution */
      double sqrt_D = Math.sqrt(D);
      double u =  cbrt(sqrt_D - q);
      double v = -cbrt(sqrt_D + q);
      
      res[0] = u + v;
      num = 1;
    }

    /* resubstitute */
    double sub = 1.0/3 * A;
    for (int i=0; i<num; i++)
      res[i] -= sub;
    
    return num;
  }
  // helper functions for the cubic solver.
  private static double cbrt(double x) {
    // XXX: standard library should provide a cube root function!
    // FURTHERMORE the standard library routine is broken for determining
    // the cube roots of negative numbers!
    if (x<0) return -Math.pow(-x, 1.0/3.0);
    return Math.pow(x, 1.0/3.0);
  }
  // how close to zero is "close enough"?
  private static final double epsilon = 1e-9;
  private static boolean isZero(double x) {
    return x > -epsilon && x < epsilon;
  }
  /**
   * Return true if some edge in this GeneralPath intersects the given
   * line.  Otherwise, return false.
   * <p>
   * GNU extension. */
  public boolean intersectsLine(Line2D l) {
    return intersectsLine(l.getX1(), l.getY1(), l.getX2(), l.getY2());
  }
  /**
   * Return true if this curve intersects the line from (x1, y1) to (x2, y2).
   * Otherwise, return false.
   * <p>
   * GNU extension. */
  public boolean intersectsLine(double x1, double y1, double x2, double y2) {
    Line2D testline = new Line2D.Double(x1, y1, x2, y2);
    // quickest test:
    if (!getBounds2D().intersects(testline.getBounds2D()))
      return false; // bounding boxes don't intersect.
    // quick test:
    GeneralPath cp = getControlPolygon();
    if (!cp.intersectsLine(testline) && !cp.contains(x1, y1))
      // if line is not contained within the control polygon and it
      // doesn't intersect any edge of the control polygon, then it doesn't
      // intersect the curve.
      return false;
    // accurate test.
    // note that a line is a degree-1 bezier curve.
    return curvesIntersect(new double[] { getX1(), getY1(),
					  getCtrlX1(), getCtrlY1(),
					  getCtrlY1(), getCtrlY2(),
					  getY1(), getY2() }, 3,
			   new double[] { x1, y1, x2, y2 }, 1);
  }
  /** GNU extension. */
  public boolean intersectsCurve(QuadCurve2D c) {
    return curvesIntersect(new double[] { getX1(), getY1(),
					  getCtrlX1(), getCtrlY1(),
					  getCtrlX2(), getCtrlY2(),
					  getX2(), getY2() }, 3,
			   new double[] { c.getX1(), c.getY1(),
					  c.getCtrlX(), c.getCtrlY(),
					  c.getX2(), c.getY2() }, 3);
  }
  /** GNU extension. */
  public boolean intersectsCurve(CubicCurve2D c) {
    return curvesIntersect(new double[] { getX1(), getY1(),
					  getCtrlX1(), getCtrlY1(),
					  getCtrlX2(), getCtrlY2(),
					  getX2(), getY2() }, 3,
			   new double[] { c.getX1(), c.getY1(),
					  c.getCtrlX1(), c.getCtrlY1(),
					  c.getCtrlX2(), c.getCtrlY2(),
					  c.getX2(), c.getY2() }, 3);
  }

  /**
   * Return true iff the given bezier curves intersect.
   * The bezier arrays should each contain (degree+1) points (pairs of
   * coordinates).
   * <p>
   * GNU extension. */
  public static boolean curvesIntersect(double[] bezier1, int degree1,
					double[] bezier2, int degree2) {
    // Is there any solution to BZ1(t1) = BZ2(t2) in both x and y?
    return _curvesIntersect(bezier1, degree1, bezier2, degree2, 0);
  }
  private static boolean _curvesIntersect(double[] bz1, int degree1,
					  double[] bz2, int degree2,
					  int depth) {
    assert bz1.length == 2*(degree1+1);
    assert bz2.length == 2*(degree2+1);
    // first approximation: curves don't intersect if their control
    // polygons don't intersect.
    // use bounding boxes as approximation to control polygons.
    double[] min = { bz1[0], bz1[1], bz2[0], bz2[1] };
    double[] max = { bz1[0], bz1[1], bz2[0], bz2[1] };
    for (int i=1; i<=degree1; i++) {
      if (bz1[2*i+0] < min[0]) min[0] = bz1[2*i+0];
      if (bz1[2*i+1] < min[1]) min[1] = bz1[2*i+1];
      if (bz1[2*i+0] > max[0]) max[0] = bz1[2*i+0];
      if (bz1[2*i+1] > max[1]) max[1] = bz1[2*i+1];
    }
    for (int i=1; i<=degree2; i++) {
      if (bz2[2*i+0] < min[2]) min[2] = bz2[2*i+0];
      if (bz2[2*i+1] < min[3]) min[3] = bz2[2*i+1];
      if (bz2[2*i+0] > max[2]) max[2] = bz2[2*i+0];
      if (bz2[2*i+1] > max[3]) max[3] = bz2[2*i+1];
    }
    if (min[0] > max[2] || min[2] > max[0] ||
	min[1] > max[3] || min[3] > max[1] )
      return false; // curves don't intersect!
    
    if (Line2D.linesIntersect
	(bz1[0], bz1[1], bz1[2*degree1], bz1[2*degree1+1],
	 bz2[0], bz2[1], bz2[2*degree2], bz2[2*degree2+1])) {
      GeneralPath gp = getControlPolygon(bz1, degree1);
      if (!gp.contains(bz2[0], bz2[1]) &&
	  !gp.contains(bz2[2*degree2], bz2[2*degree2+1]))
	return true; // beziers must cross each other.
      gp = getControlPolygon(bz2, degree2);
      if (!gp.contains(bz1[0], bz1[1]) &&
	  !gp.contains(bz1[2*degree1], bz1[2*degree1+1]))
	return true; // beziers must cross each other.
    }

    if (depth++ > MAX_RECURSION_DEPTH) // limit recursion in worst case.
      return Line2D.linesIntersect
	(bz1[0], bz1[1], bz1[2*degree1], bz1[2*degree1+1],
	 bz2[0], bz2[1], bz2[2*degree2], bz2[2*degree2+1]);
    // otherwise, subdivide the curviest curve and recurse.
    double nonlinearity1 = 0, nonlinearity2 = 0;
    for (int i=1; i<degree1; i++)
      nonlinearity1 = Math.max
	(nonlinearity1, Line2D.ptSegDistSq(bz1[0], bz1[1],
					   bz1[2*degree1], bz1[2*degree1+1],
					   bz1[2*i], bz1[2*i+1]));
    for (int i=1; i<degree2; i++)
      nonlinearity2= Math.max
	(nonlinearity2, Line2D.ptSegDistSq(bz2[0], bz2[1],
					   bz2[2*degree2], bz2[2*degree2+1],
					   bz2[2*i], bz2[2*i+1]));
    if (nonlinearity2 > nonlinearity1) {
      double[] t1 = bz1; bz1 = bz2; bz2 = t1;
      int t2 = degree1; degree1 = degree2; degree2 = t2;
    }
    // subdivide bz1.
    double[] bz1l = new double[bz1.length], bz1r = new double[bz1.length];
    subdivide(bz1, 0, bz1l, 0, bz1r, 0, .5, degree1, 2); // x of bz1
    subdivide(bz1, 1, bz1l, 1, bz1r, 1, .5, degree1, 2); // y of bz1

    return
      _curvesIntersect(bz1l, degree1, bz2, degree2, depth) ||
      _curvesIntersect(bz1r, degree1, bz2, degree2, depth);
  }


  public boolean contains(double x, double y)
  {
    // true iff <x,y> is inside the polygon created by closing this curve.
    GeneralPath gp = new GeneralPath(this);
    gp.closePath();
    return gp.contains(x,y);
  }
  public boolean contains(Point2D p)
  {
    return contains(p.getX(), p.getY());
  }
  public boolean intersects(double x, double y, double w, double h)
  {
    // conservatively approximate by intersecting box with control polygon.
    // use polygon intersection code in GeneralPath.
    return getControlPolygon().intersects(x,y,w,h);
  }
  public boolean intersects(Rectangle2D r)
  {
    return intersects(r.getX(), r.getY(), r.getWidth(), r.getHeight());
  }
  public boolean contains(double x, double y, double w, double h)
  {
    // true iff box is inside the polygon created by closing this curve.
    GeneralPath gp = new GeneralPath(this);
    gp.closePath();
    return gp.contains(x,y,w,h);
  }
  public boolean contains(Rectangle2D r)
  {
    return contains(r.getX(), r.getY(), r.getWidth(), r.getHeight());
  }
  public Rectangle getBounds()
  {
    return getBounds2D().getBounds();
  }
  /** GNU extension. */
  public GeneralPath getControlPolygon() {
    return getControlPolygon(new double[] { getX1(), getY1(),
					    getCtrlX1(), getCtrlY1(),
					    getCtrlX2(), getCtrlY2(),
					    getX2(), getY2() }, 3);
  }
  /** GNU extension. */
  public static GeneralPath getControlPolygon(double bezier[], int degree) {
    GeneralPath gp = new GeneralPath(GeneralPath.WIND_EVEN_ODD);
    gp.moveTo(bezier[0], bezier[1]);
    for (int i=0; i<=degree; i++)
      gp.lineTo(bezier[2*i], bezier[2*i+1]);
    gp.closePath();
    return gp;
  }

  public PathIterator getPathIterator(final AffineTransform at)
  {
    return new PathIterator()
    {
      /** Current coordinate. */
      private int current;

      public int getWindingRule()
      {
        return WIND_NON_ZERO;
      }

      public boolean isDone()
      {
        return current >= 2;
      }

      public void next()
      {
        current++;
      }

      public int currentSegment(float[] coords)
      {
        if (current == 0)
          {
            coords[0] = (float) getX1();
            coords[1] = (float) getY1();
            if (at != null)
              at.transform(coords, 0, coords, 0, 1);
            return SEG_MOVETO;
          }
        if (current == 1)
          {
            coords[0] = (float) getCtrlX1();
            coords[1] = (float) getCtrlY1();
            coords[2] = (float) getCtrlX2();
            coords[3] = (float) getCtrlY2();
            coords[4] = (float) getX2();
            coords[5] = (float) getY2();
            if (at != null)
              at.transform(coords, 0, coords, 0, 3);
            return SEG_CUBICTO;
          }
        throw new NoSuchElementException("cubic iterator out of bounds");
      }

      public int currentSegment(double[] coords)
      {
        if (current == 0)
          {
            coords[0] = getX1();
            coords[1] = getY1();
            if (at != null)
              at.transform(coords, 0, coords, 0, 1);
            return SEG_MOVETO;
          }
        if (current == 1)
          {
            coords[0] = getCtrlX1();
            coords[1] = getCtrlY1();
            coords[2] = getCtrlX2();
            coords[3] = getCtrlY2();
            coords[4] = getX2();
            coords[5] = getY2();
            if (at != null)
              at.transform(coords, 0, coords, 0, 3);
            return SEG_CUBICTO;
          }
        throw new NoSuchElementException("cubic iterator out of bounds");
      }
    };
  }
  public PathIterator getPathIterator(AffineTransform at, double flatness)
  {
    return new FlatteningPathIterator(getPathIterator(at), flatness);
  }

  /**
   * Create a new curve of the same run-time type with the same contents as
   * this one.
   *
   * @return the clone
   */
  public Object clone()
  {
    try
      {
        return super.clone();
      }
    catch (CloneNotSupportedException e)
      {
        throw (Error) new InternalError().initCause(e); // Impossible
      }
  }

  public static class Double extends CubicCurve2D
  {
    public double x1;
    public double y1;
    public double ctrlx1;
    public double ctrly1;
    public double ctrlx2;
    public double ctrly2;
    public double x2;
    public double y2;

    public Double()
    {
    }

    public Double(double x1, double y1, double cx1, double cy1,
                  double cx2, double cy2, double x2, double y2)
    {
      this.x1 = x1;
      this.y1 = y1;
      ctrlx1 = cx1;
      ctrly1 = cy1;
      ctrlx2 = cx2;
      ctrly2 = cy2;
      this.x2 = x2;
      this.y2 = y2;
    }
    /** GNU extension. */
    public Double(QuadCurve2D quad) {
      double[] pts = new double[] { quad.getX1(), quad.getY1(),
				    quad.getCtrlX(), quad.getCtrlY(),
				    quad.getX2(), quad.getY2(),
				    0,0 /* extra space for cubic */ };
      raiseDegree(pts, 2, pts, 3);
      this.x1 = pts[0];
      this.y1 = pts[1];
      this.ctrlx1 = pts[2];
      this.ctrly1 = pts[3];
      this.ctrlx2 = pts[4];
      this.ctrly2 = pts[5];
      this.x2 = pts[6];
      this.y2 = pts[7];
    }

    public double getX1()
    {
      return x1;
    }
    public double getY1()
    {
      return y1;
    }
    public Point2D getP1()
    {
      return new Point2D.Double(x1, y1);
    }

    public double getCtrlX1()
    {
      return ctrlx1;
    }
    public double getCtrlY1()
    {
      return ctrly1;
    }
    public Point2D getCtrlP1()
    {
      return new Point2D.Double(ctrlx1, ctrly1);
    }

    public double getCtrlX2()
    {
      return ctrlx2;
    }
    public double getCtrlY2()
    {
      return ctrly2;
    }
    public Point2D getCtrlP2()
    {
      return new Point2D.Double(ctrlx2, ctrly2);
    }

    public double getX2()
    {
      return x2;
    }
    public double getY2()
    {
      return y2;
    }
    public Point2D getP2()
    {
      return new Point2D.Double(x2, y2);
    }

    public void setCurve(double x1, double y1, double cx1, double cy1,
                         double cx2, double cy2, double x2, double y2)
    {
      this.x1 = x1;
      this.y1 = y1;
      ctrlx1 = cx1;
      ctrly1 = cy1;
      ctrlx2 = cx2;
      ctrly2 = cy2;
      this.x2 = x2;
      this.y2 = y2;
    }
    public Rectangle2D getBounds2D()
    {
      double nx1 = Math.min(Math.min(x1, ctrlx1), Math.min(ctrlx2, x2));
      double ny1 = Math.min(Math.min(y1, ctrly1), Math.min(ctrly2, y2));
      double nx2 = Math.max(Math.max(x1, ctrlx1), Math.max(ctrlx2, x2));
      double ny2 = Math.max(Math.max(y1, ctrly1), Math.max(ctrly2, y2));
      return new Rectangle2D.Double(nx1, ny1, nx2 - nx1, ny2 - ny1);
    }
  } // class Double

  public static class Float extends CubicCurve2D
  {
    public float x1;
    public float y1;
    public float ctrlx1;
    public float ctrly1;
    public float ctrlx2;
    public float ctrly2;
    public float x2;
    public float y2;

    public Float()
    {
    }

    public Float(float x1, float y1, float cx1, float cy1,
                 float cx2, float cy2, float x2, float y2)
    {
      this.x1 = x1;
      this.y1 = y1;
      ctrlx1 = cx1;
      ctrly1 = cy1;
      ctrlx2 = cx2;
      ctrly2 = cy2;
      this.x2 = x2;
      this.y2 = y2;
    }
    /** GNU extension. */
    public Float(QuadCurve2D quad) {
      double[] pts = new double[] { quad.getX1(), quad.getY1(),
				    quad.getCtrlX(), quad.getCtrlY(),
				    quad.getX2(), quad.getY2(),
				    0,0 /* extra space for cubic */ };
      raiseDegree(pts, 2, pts, 3);
      this.x1 = (float) pts[0];
      this.y1 = (float) pts[1];
      this.ctrlx1 = (float) pts[2];
      this.ctrly1 = (float) pts[3];
      this.ctrlx2 = (float) pts[4];
      this.ctrly2 = (float) pts[5];
      this.x2 = (float) pts[6];
      this.y2 = (float) pts[7];
    }

    public double getX1()
    {
      return x1;
    }
    public double getY1()
    {
      return y1;
    }
    public Point2D getP1()
    {
      return new Point2D.Float(x1, y1);
    }

    public double getCtrlX1()
    {
      return ctrlx1;
    }
    public double getCtrlY1()
    {
      return ctrly1;
    }
    public Point2D getCtrlP1()
    {
      return new Point2D.Float(ctrlx1, ctrly1);
    }

    public double getCtrlX2()
    {
      return ctrlx2;
    }
    public double getCtrlY2()
    {
      return ctrly2;
    }
    public Point2D getCtrlP2()
    {
      return new Point2D.Float(ctrlx2, ctrly2);
    }

    public double getX2()
    {
      return x2;
    }
    public double getY2()
    {
      return y2;
    }
    public Point2D getP2()
    {
      return new Point2D.Float(x2, y2);
    }

    public void setCurve(double x1, double y1, double cx1, double cy1,
                         double cx2, double cy2, double x2, double y2)
    {
      this.x1 = (float) x1;
      this.y1 = (float) y1;
      ctrlx1 = (float) cx1;
      ctrly1 = (float) cy1;
      ctrlx2 = (float) cx2;
      ctrly2 = (float) cy2;
      this.x2 = (float) x2;
      this.y2 = (float) y2;
    }
    public void setCurve(float x1, float y1, float cx1, float cy1,
                         float cx2, float cy2, float x2, float y2)
    {
      this.x1 = x1;
      this.y1 = y1;
      ctrlx1 = cx1;
      ctrly1 = cy1;
      ctrlx2 = cx2;
      ctrly2 = cy2;
      this.x2 = x2;
      this.y2 = y2;
    }
    public Rectangle2D getBounds2D()
    {
      float nx1 = (float) Math.min(Math.min(x1, ctrlx1), Math.min(ctrlx2, x2));
      float ny1 = (float) Math.min(Math.min(y1, ctrly1), Math.min(ctrly2, y2));
      float nx2 = (float) Math.max(Math.max(x1, ctrlx1), Math.max(ctrlx2, x2));
      float ny2 = (float) Math.max(Math.max(y1, ctrly1), Math.max(ctrly2, y2));
      return new Rectangle2D.Float(nx1, ny1, nx2 - nx1, ny2 - ny1);
    }
  } // class Float
} // class CubicCurve2D
