package net.cscott.mallard;

import net.cscott.pcbmill.*;

import gnu.getopt.*;

import java.awt.geom.*;
import java.io.File;
import java.io.FileWriter;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.*;

// for test applet
import java.awt.*;
import java.awt.image.*;
import javax.imageio.*;
import javax.swing.*;

import gnu.java.awt.geom.GeneralPath;
import gnu.java.awt.geom.CubicCurve2D;

/* Goal of this routine is to polygonize and translate the SVG input.
 * We also have to identify holes.
 */
/* Description of output file:

  .node files:
    First line:  <# of vertices> <dimension (must be 2)> <# of attributes>
                                           <# of boundary markers (0 or 1)>
    Remaining lines:  <vertex #> <x> <y> [attributes] [boundary marker]

    The attributes, which are typically floating-point values of physical
    quantities (such as mass or conductivity) associated with the nodes of
    a finite element mesh, are copied unchanged to the output mesh.  If -q,
    -a, -u, or -s is selected, each new Steiner point added to the mesh
    has attributes assigned to it by linear interpolation.

    If the fourth entry of the first line is `1', the last column of the
    remainder of the file is assumed to contain boundary markers.  Boundary
    markers are used to identify boundary vertices and vertices resting on
    PSLG segments; a complete description appears in a section below.  The
    .node file produced by Triangle contains boundary markers in the last
    column unless they are suppressed by the -B switch.

  .ele files:
    First line:  <# of triangles> <nodes per triangle> <# of attributes>
    Remaining lines:  <triangle #> <node> <node> <node> ... [attributes]

    Nodes are indices into the corresponding .node file.  The first three
    nodes are the corner vertices, and are listed in counterclockwise order
    around each triangle.  (The remaining nodes, if any, depend on the type
    of finite element used.)

    The attributes are just like those of .node files.  Because there is no
    simple mapping from input to output triangles, an attempt is made to
    interpolate attributes, which may result in a good deal of diffusion of
    attributes among nearby triangles as the triangulation is refined.
    Attributes do not diffuse across segments, so attributes used to
    identify segment-bounded regions remain intact.

    In .ele files produced by Triangle, each triangular element has three
    nodes (vertices) unless the -o2 switch is used, in which case
    subparametric quadratic elements with six nodes each are generated.
    The first three nodes are the corners in counterclockwise order, and
    the fourth, fifth, and sixth nodes lie on the midpoints of the edges
    opposite the first, second, and third vertices, respectively.

  .poly files:
    First line:  <# of vertices> <dimension (must be 2)> <# of attributes>
                                           <# of boundary markers (0 or 1)>
    Following lines:  <vertex #> <x> <y> [attributes] [boundary marker]
    One line:  <# of segments> <# of boundary markers (0 or 1)>
    Following lines:  <segment #> <endpoint> <endpoint> [boundary marker]
    One line:  <# of holes>
    Following lines:  <hole #> <x> <y>
    Optional line:  <# of regional attributes and/or area constraints>
    Optional following lines:  <region #> <x> <y> <attribute> <max area>

    A .poly file represents a PSLG, as well as some additional information.
    The first section lists all the vertices, and is identical to the
    format of .node files.  <# of vertices> may be set to zero to indicate
    that the vertices are listed in a separate .node file; .poly files
    produced by Triangle always have this format.  A vertex set represented
    this way has the advantage that it may easily be triangulated with or
    without segments (depending on whether the .poly or .node file is
    read).

    The second section lists the segments.  Segments are edges whose
    presence in the triangulation is enforced (although each segment may be
    subdivided into smaller edges).  Each segment is specified by listing
    the indices of its two endpoints.  This means that you must include its
    endpoints in the vertex list.  Each segment, like each point, may have
    a boundary marker.

    If -q, -a, -u, and -s are not selected, Triangle produces a constrained
    Delaunay triangulation (CDT), in which each segment appears as a single
    edge in the triangulation.  If -q, -a, -u, or -s is selected, Triangle
    produces a conforming constrained Delaunay triangulation (CCDT), in
    which segments may be subdivided into smaller edges.  If -L is selected
    as well, Triangle produces a conforming Delaunay triangulation, so
    every triangle is Delaunay, and not just constrained Delaunay.

    The third section lists holes (and concavities, if -c is selected) in
    the triangulation.  Holes are specified by identifying a point inside
    each hole.  After the triangulation is formed, Triangle creates holes
    by eating triangles, spreading out from each hole point until its
    progress is blocked by PSLG segments; you must be careful to enclose
    each hole in segments, or your whole triangulation might be eaten away.
    If the two triangles abutting a segment are eaten, the segment itself
    is also eaten.  Do not place a hole directly on a segment; if you do,
    Triangle chooses one side of the segment arbitrarily.

    The optional fourth section lists regional attributes (to be assigned
    to all triangles in a region) and regional constraints on the maximum
    triangle area.  Triangle reads this section only if the -A switch is
    used or the -a switch is used without a number following it, and the -r
    switch is not used.  Regional attributes and area constraints are
    propagated in the same manner as holes; you specify a point for each
    attribute and/or constraint, and the attribute and/or constraint
    affects the whole region (bounded by segments) containing the point.
    If two values are written on a line after the x and y coordinate, the
    first such value is assumed to be a regional attribute (but is only
    applied if the -A switch is selected), and the second value is assumed
    to be a regional area constraint (but is only applied if the -a switch
    is selected).  You may specify just one value after the coordinates,
    which can serve as both an attribute and an area constraint, depending
    on the choice of switches.  If you are using the -A and -a switches
    simultaneously and wish to assign an attribute to some region without
    imposing an area constraint, use a negative maximum area.

    When a triangulation is created from a .poly file, you must either
    enclose the entire region to be triangulated in PSLG segments, or
    use the -c switch, which encloses the convex hull of the input vertex
    set.  If you do not use the -c switch, Triangle eats all triangles that
    are not enclosed by segments; if you are not careful, your whole
    triangulation may be eaten away.  If you do use the -c switch, you can
    still produce concavities by the appropriate placement of holes just
    within the convex hull.

    An ideal PSLG has no intersecting segments, nor any vertices that lie
    upon segments (except, of course, the endpoints of each segment.)  You
    aren't required to make your .poly files ideal, but you should be aware
    of what can go wrong.  Segment intersections are relatively safe--
    Triangle calculates the intersection points for you and adds them to
    the triangulation--as long as your machine's floating-point precision
    doesn't become a problem.  You are tempting the fates if you have three
    segments that cross at the same location, and expect Triangle to figure
    out where the intersection point is.  Thanks to floating-point roundoff
    error, Triangle will probably decide that the three segments intersect
    at three different points, and you will find a minuscule triangle in
    your output--unless Triangle tries to refine the tiny triangle, uses
    up the last bit of machine precision, and fails to terminate at all.
    You're better off putting the intersection point in the input files,
    and manually breaking up each segment into two.  Similarly, if you
    place a vertex at the middle of a segment, and hope that Triangle will
    break up the segment at that vertex, you might get lucky.  On the other
    hand, Triangle might decide that the vertex doesn't lie precisely on
    the segment, and you'll have a needle-sharp triangle in your output--or
    a lot of tiny triangles if you're generating a quality mesh.

    When Triangle reads a .poly file, it also writes a .poly file, which
    includes all edges that are parts of input segments.  If the -c switch
    is used, the output .poly file also includes all of the edges on the
    convex hull.  Hence, the output .poly file is useful for finding edges
    associated with input segments and for setting boundary conditions in
    finite element simulations.  Moreover, you will need it if you plan to
    refine the output mesh, and don't want segments to be missing in later
    triangulations.

  .area files:
    First line:  <# of triangles>
    Following lines:  <triangle #> <maximum area>

    An .area file associates with each triangle a maximum area that is used
    for mesh refinement.  As with other file formats, every triangle must
    be represented, and they must be numbered consecutively.  A triangle
    may be left unconstrained by assigning it a negative maximum area.

  .edge files:
    First line:  <# of edges> <# of boundary markers (0 or 1)>
    Following lines:  <edge #> <endpoint> <endpoint> [boundary marker]

    Endpoints are indices into the corresponding .node file.  Triangle can
    produce .edge files (use the -e switch), but cannot read them.  The
    optional column of boundary markers is suppressed by the -B switch.

    In Voronoi diagrams, one also finds a special kind of edge that is an
    infinite ray with only one endpoint.  For these edges, a different
    format is used:

        <edge #> <endpoint> -1 <direction x> <direction y>

    The `direction' is a floating-point vector that indicates the direction
    of the infinite ray.

  .neigh files:
    First line:  <# of triangles> <# of neighbors per triangle (always 3)>
    Following lines:  <triangle #> <neighbor> <neighbor> <neighbor>

    Neighbors are indices into the corresponding .ele file.  An index of -1
    indicates no neighbor (because the triangle is on an exterior
    boundary).  The first neighbor of triangle i is opposite the first
    corner of triangle i, and so on.

    Triangle can produce .neigh files (use the -n switch), but cannot read
    them.

Boundary Markers:

  Boundary markers are tags used mainly to identify which output vertices
  and edges are associated with which PSLG segment, and to identify which
  vertices and edges occur on a boundary of the triangulation.  A common
  use is to determine where boundary conditions should be applied to a
  finite element mesh.  You can prevent boundary markers from being written
  into files produced by Triangle by using the -B switch.

  The boundary marker associated with each segment in an output .poly file
  and each edge in an output .edge file is chosen as follows:
    - If an output edge is part or all of a PSLG segment with a nonzero
      boundary marker, then the edge is assigned the same marker.
    - Otherwise, if the edge occurs on a boundary of the triangulation
      (including boundaries of holes), then the edge is assigned the marker
      one (1).
    - Otherwise, the edge is assigned the marker zero (0).
  The boundary marker associated with each vertex in an output .node file
  is chosen as follows:
    - If a vertex is assigned a nonzero boundary marker in the input file,
      then it is assigned the same marker in the output .node file.
    - Otherwise, if the vertex lies on a PSLG segment (including the
      segment's endpoints) with a nonzero boundary marker, then the vertex
      is assigned the same marker.  If the vertex lies on several such
      segments, one of the markers is chosen arbitrarily.
    - Otherwise, if the vertex occurs on a boundary of the triangulation,
      then the vertex is assigned the marker one (1).
    - Otherwise, the vertex is assigned the marker zero (0).

  If you want Triangle to determine for you which vertices and edges are on
  the boundary, assign them the boundary marker zero (or use no markers at
  all) in your input files.  In the output files, all boundary vertices,
  edges, and segments are assigned the value one.
*/

public class Contour {
    private static final String PROGNAME="Contour";

    static double FLATNESS=.72;
    static double MINLENGTH=.00072;
    static double STEPSIZE=5; // 1 point every five original pixels.
    static String colorImageFilename=null;
    static boolean DO_DISPLAY=false;
    static boolean DO_VERBOSE_DISPLAY=false;
    static Rectangle2D ZOOMBOX=null;

    public static void main(String[] args) throws java.io.IOException {
	Getopt g = new Getopt(PROGNAME, args, "hi:z:dW;", new LongOpt[] {
	    new LongOpt("help", LongOpt.NO_ARGUMENT, null, 'h'),
	    new LongOpt("color-image", LongOpt.REQUIRED_ARGUMENT, null, 'i'),
	    new LongOpt("image", LongOpt.REQUIRED_ARGUMENT, null, 'i'),
	    new LongOpt("zoom", LongOpt.REQUIRED_ARGUMENT, null, 'z'),
	    new LongOpt("display", LongOpt.NO_ARGUMENT, null, 'd'),
	});
	int c; String arg; boolean saw_error=false;
	while ((c = g.getopt()) != -1)
	    switch(c) {
	    case 'h':
		saw_error=true; // forces display of help.
		break;
	    case 'i':
		// original color image.
		colorImageFilename = g.getOptarg();
		break;
	    case 'z':
		// zoom the display.
		ZOOMBOX = grokbox(g.getOptarg());
		if (ZOOMBOX==null) saw_error=true;
		else // flip on y-axis.
		    ZOOMBOX=new Rectangle2D.Double
			(ZOOMBOX.getX(), -ZOOMBOX.getMaxY(),
			 ZOOMBOX.getWidth(), ZOOMBOX.getHeight());
		break;
	    case 'd': // double d gives verbose display.
		if (DO_DISPLAY) DO_VERBOSE_DISPLAY=true;
		DO_DISPLAY = true;
		break;
	    case '?':
		// already printed error msg.
	    default:
		saw_error=true;
		break;
	    }
	if (g.getOptind()+2 != args.length) {
	    System.err.println("Too few or too many arguments.");
	    saw_error=true;
	}
	if (saw_error) {
	    System.err.println("Usage: "+
			       PROGNAME+" [-hi_d] <infile> <outfile>");
	    System.exit(1);
	}
	String infile = args[g.getOptind()];
	String outfile = args[g.getOptind()+1];

	GeneralPath path = new SVGParser(new File(infile)).image;
	// undo svg scaling so that 1 unit == 1 pixel again.
	path.transform(AffineTransform.getScaleInstance(72,72));

	if (false) // override with very simple path for testing.
	    path = testPath();

	// voronoi diagram only approximates the medial axis if the boundary
	// is sampled by equally-spaced points.  Make an attempt here to
	// regularize the point spacing in order to improve our medial
	// axis.  We do this by subdividing long segments.
	// Note that we do this *before* curves are flattened, for better
	// point placement.
	path = choppedPath(path);

	// Flatten contour, and remove too-small segments.
	Map<Point2D,Integer> contourIndex=new LinkedHashMap<Point2D,Integer>();
	List<Line2D> contourEdges = new ArrayList<Line2D>();
	GeneralPath contourPath = new GeneralPath();

	double[] coords = new double[6];
	double startx=0, starty=0, lastx=0, lasty=0;
	boolean first=true;
	for (PathIterator pi = path.getPathIterator(null, FLATNESS);
	     !pi.isDone(); pi.next()) {
	    switch(pi.currentSegment(coords)) {
	    case PathIterator.SEG_MOVETO:
		startx = coords[0]; starty = coords[1];
		lastx = coords[0]; lasty = coords[1];
		first = true;
		break;
	    case PathIterator.SEG_CLOSE:
		coords[0] = startx; coords[1] = starty;
		// fall through.
	    case PathIterator.SEG_LINETO:
		if (first) {
		    first=false;
		    addIfNew(contourIndex, lastx, lasty);
		    contourPath.moveTo(lastx, lasty);
		}
		Line2D.Double l = new Line2D.Double
		    (lastx, lasty, coords[0], coords[1]);
		if (l.getP1().distance(l.getP2()) > MINLENGTH) {
		    addIfNew(contourIndex, coords[0], coords[1]);
		    contourPath.lineTo(coords[0], coords[1]);
		    contourEdges.add(l);
		    lastx = coords[0]; lasty = coords[1];
		}
		break;
	    case PathIterator.SEG_QUADTO:
	    case PathIterator.SEG_CUBICTO:
	    default:
		assert false; // should be no curves left.
	    }
	}

	// now emit .poly file in appropriate format.
	PrintWriter pw = new PrintWriter(new FileWriter(outfile));
	//   vertices
	pw.println(contourIndex.size()+" 2 0 0");
	int i=0;
	for (Point2D pt: contourIndex.keySet()) {
	    assert i==contourIndex.get(pt);
	    pw.println((i++)+" "+pt.getX()+" "+pt.getY());
	}
	//   segments
	pw.println(contourEdges.size()+" 1");
	i=0;
	for (Line2D line: contourEdges) {
	    int pt1num = contourIndex.get(line.getP1());
	    int pt2num = contourIndex.get(line.getP2());
	    pw.println((i++)+" "+pt1num+" "+pt2num+" 1"); // boundary marker
	}
	//   holes
	pw.println("0");
	//   attributes or area contraints
	pw.println("0");
	pw.close();

	if (DO_DISPLAY)
	    display(contourIndex, contourPath);
    }

    private static void addIfNew(Map<Point2D,Integer> contourIndex,
				 double x, double y) {
	Point2D pt = new Point2D.Double(x, y);
	if (contourIndex.containsKey(pt)) return;
	contourIndex.put(pt, new Integer(contourIndex.size()));
    }

    private static GeneralPath choppedPath(Shape s) {
	GeneralPath result = new GeneralPath();
	double[] coords = new double[6];
	double startx=0, starty=0, lastx=0, lasty=0;
	boolean first=true, doClose=false;
	for (PathIterator pi = s.getPathIterator(null);
	     !pi.isDone(); pi.next()) {
	    switch(pi.currentSegment(coords)) {
	    case PathIterator.SEG_MOVETO:
		startx = coords[0]; starty = coords[1];
		lastx = coords[0]; lasty = coords[1];
		first = true;
		break;
	    case PathIterator.SEG_CLOSE:
		coords[0] = startx; coords[1] = starty;
		doClose=true;
		// fall through.
	    case PathIterator.SEG_LINETO:
		if (first) {
		    first=false;
		    result.moveTo(lastx, lasty);
		}
		doLine(result, lastx, lasty, coords[0], coords[1], doClose);
		lastx = coords[0]; lasty = coords[1]; doClose=false;
		break;
	    case PathIterator.SEG_QUADTO:
		// raise quad curve to a cubic and use SEG_CUBICTO
		double[] param = new double[8];
		param[0] = lastx; param[1]=lasty;
		System.arraycopy(coords, 0, param, 2, 4);
		CubicCurve2D.raiseDegree(param, 2, param, 3);
		System.arraycopy(param, 2, coords, 0, 6);
		// fall through
	    case PathIterator.SEG_CUBICTO:
		if (first) {
		    first=false;
		    result.moveTo(lastx, lasty);
		}
		doQuad(result, lastx, lasty, coords[0], coords[1],
		       coords[2], coords[3], coords[4], coords[5]);
		lastx = coords[4]; lasty = coords[5];
		break;
	    default:
		assert false;
	    }
	}
	return result;
    }
    // subdivide a line.
    private static void doLine(GeneralPath result,
			       double x1, double y1, double x2, double y2,
			       boolean doClose) {
	double dist = Point2D.distance(x1, y1, x2, y2);
	if (dist <= STEPSIZE) {
	    if (doClose) result.closePath();
	    else result.lineTo(x2, y2);
	} else {
	    double midx = (x1+x2)/2, midy = (y1+y2)/2;
	    doLine(result, x1, y1, midx, midy, false);
	    doLine(result, midx, midy, x2, y2, doClose);
	}
    }
    // subdivide a curve.
    private static void doQuad(GeneralPath result,
			       double x1, double y1, double x2, double y2,
			       double x3, double y3, double x4, double y4) {
	// approximate quad length with perimeter of control polygon
	// this is an upper bound.
	double dist =
	    Point2D.distance(x1, y1, x2, y2) +
	    Point2D.distance(x2, y2, x3, y3) +
	    Point2D.distance(x3, y3, x4, y4);
	if (dist <= STEPSIZE) {
	    result.curveTo(x2, y2, x3, y3, x4, y4);
	} else {
	    double[] d = new double[] { x1,y1,x2,y2,x3,y3,x4,y4,
					0,0,0,0,0,0,0,0 };
	    CubicCurve2D.subdivide(d, 0, d, 0, d, 8, 0.5);
	    doQuad(result, d[0], d[1], d[2], d[3], d[4], d[5], d[6], d[7]);
	    doQuad(result, d[8], d[9],d[10],d[11],d[12],d[13],d[14],d[15]);
	}
    }

    /** Parse a string in the form <float>x<float>+<float>+<float>;
     *  returns a box of the specified width, height, and origin.
     *  If the origin is unspecified, it defaults to 0,0.  If the
     *  height is unspecified, it defaults to the width. */
    private static Rectangle2D grokbox(String s) {
	Matcher m = BOX_PATTERN.matcher(s);
	if (!m.matches()) {
	    System.err.println("Bad box specification: "+s);
	    return null;
	}
	String width=m.group(1);
	String height=m.group(2);
	String x=m.group(3);
	String y=m.group(4);
	if (height==null) height=width;
	if (x==null) x="0";
	if (y==null) y="0";
	return new Rectangle2D.Double
	    (Double.parseDouble(x), Double.parseDouble(y),
	     Double.parseDouble(width), Double.parseDouble(height));
    }
    static final Pattern BOX_PATTERN;
    static {
	String FLOAT="(?:(?:[0-9]+(?:[.][0-9]*)?)|(?:[.][0-9]+))";
	BOX_PATTERN = Pattern.compile
	    ("\\A"+
	     "("+FLOAT+")(?:x("+FLOAT+"))?(?:([+-]"+FLOAT+")([+-]"+FLOAT+")?)?"
	     +"\\Z"
	     );
    }
    // testing
    static void display(Map<Point2D,Integer> contourIndex,
			GeneralPath contourPath) 
	throws java.io.IOException {
	BufferedImage colorImage=null; AffineTransform colorImageXF=null;
	if (colorImageFilename!=null) {
	    colorImage = ImageIO.read(new File(colorImageFilename));
	    // image is flipped over y axis compared to everything else.
	    colorImageXF = AffineTransform.getScaleInstance(1,-1);
	}
	
        JFrame f = new JFrame(PROGNAME);
        f.addWindowListener(new java.awt.event.WindowAdapter() {
		public void windowClosing(java.awt.event.WindowEvent e) {
		    System.exit(0);
		}
	    });
        JApplet applet = new App(contourIndex, contourPath,
				 colorImage, colorImageXF);
        f.getContentPane().add("Center", applet);
        applet.init();
        f.pack();
        f.setSize(new Dimension(700,700));
        f.show();	
    }
    static class App extends javax.swing.JApplet {
	final Map<Point2D,Integer> contourIndex;
	final GeneralPath contourPath;
	final Point2D minpt, maxpt;
	final BufferedImage colorImage;
	final AffineTransform colorImageXF;
	final static Color bg = Color.white;
	final static Color fg = Color.black;

	App(Map<Point2D,Integer> contourIndex,
	    GeneralPath contourPath,
	    BufferedImage colorImage, AffineTransform colorImageXF) {
	    this.contourIndex = contourIndex;
	    this.contourPath = contourPath;
	    this.colorImage = colorImage;
	    this.colorImageXF = colorImageXF;
	    if (ZOOMBOX==null) {
		double minx=Double.POSITIVE_INFINITY;
		double miny=Double.POSITIVE_INFINITY;
		double maxx=Double.NEGATIVE_INFINITY;
		double maxy=Double.NEGATIVE_INFINITY;
		for (Point2D pt : contourIndex.keySet()) {
		    if (pt.getX() < minx) minx = pt.getX();
		    if (pt.getY() < miny) miny = pt.getY();
		    if (pt.getX() > maxx) maxx = pt.getX();
		    if (pt.getY() > maxy) maxy = pt.getY();
		}
		this.minpt = new Point2D.Double(minx, miny);
		this.maxpt = new Point2D.Double(maxx, maxy);
	    } else {
		this.minpt = new Point2D.Double
		    (ZOOMBOX.getMinX(), ZOOMBOX.getMinY());
		this.maxpt = new Point2D.Double
		    (ZOOMBOX.getMaxX(), ZOOMBOX.getMaxY());
	    }
	}
	public void init() {
	    //Initialize drawing colors
	    setBackground(bg);
	    setForeground(fg);
	}
	
	public void paint(Graphics g) {
	    Graphics2D g2 = (Graphics2D) g;
	    g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
	    Dimension d = getSize();

	    g2.setPaint(fg);

	    // create area scaled to fit.
	    // also flip the y axis, since gerber has the origin at the lower left
	    AffineTransform t = AffineTransform.getTranslateInstance
		(-minpt.getX(), -minpt.getY());

	    t.preConcatenate
		(AffineTransform.getScaleInstance(1, -1));
	    t.preConcatenate
		(AffineTransform.getTranslateInstance
		 (0, maxpt.getY()-minpt.getY()));

	    double scaleX = (d.width-16)/(maxpt.getX()-minpt.getX());
	    double scaleY = (d.height-16)/(maxpt.getY()-minpt.getY());
	    double scale = Math.min(scaleX, scaleY);
	    t.preConcatenate
		(AffineTransform.getScaleInstance(scale,scale));
	    t.preConcatenate
		(AffineTransform.getTranslateInstance(8,8));

	    if (colorImage!=null) {
		AffineTransform at = (AffineTransform) t.clone();
		at.concatenate(colorImageXF);
		g2.drawImage(colorImage, at, null);
	    }

	    if (DO_VERBOSE_DISPLAY) {
	    Shape dot = new Ellipse2D.Double(-2,-2,4,4);

	    for (Point2D pt : contourIndex.keySet()) {
		float percent =
		    contourIndex.get(pt).floatValue() / contourIndex.size();
		g2.setPaint(Color.getHSBColor(percent,1,0.75f));
		Point2D loc = t.transform(pt, null);
		g2.fill(AffineTransform.getTranslateInstance
			(loc.getX(), loc.getY()).createTransformedShape(dot));
	    }
	    }
	    
	    g2.setPaint(Color.red);
	    g2.draw(t.createTransformedShape(contourPath));

	    Color fg3D = Color.lightGray;
	    g2.setPaint(fg3D);
	    g2.draw3DRect(0, 0, d.width - 1, d.height - 1, true);
	    g2.draw3DRect(3, 3, d.width - 7, d.height - 7, false);
	    g2.setPaint(fg);
	}
    }

    // for testing.
    private static GeneralPath testPath() {
	    GeneralPath path = new GeneralPath();
	    path.moveTo(0,0);
	    path.lineTo(15,0);
	    path.lineTo(15,25);
	    path.lineTo(0,25);
	    path.closePath();
	    path.moveTo(5,10);
	    path.lineTo(10,10);
	    path.lineTo(10,5);
	    path.closePath();
	    path.moveTo(5,15);
	    path.lineTo(5,20);
	    path.lineTo(10,15);
	    path.closePath();
	    return path;
    }
}
