package net.cscott.mallard;

import java.awt.Color;
import java.awt.color.ColorSpace;

/** <code>BaseColorSpace</code> implements some common code for
 *  extensions to <code>ColorSpace</code>. */
public abstract class BaseColorSpace extends ColorSpace {
    static final ColorSpace _CS_sRGB = ColorSpace.getInstance(CS_sRGB);
    BaseColorSpace(int type, int num_components) {
	super(type, num_components);
    }

    /** White point D50, in CIE XYZ. */
    static final double[] CIE_D50 = new double[] { .9642f, 1f, .8249f };
    static final int X=0, Y=1, Z=2;
    static final double ONE_THIRD=1/3.;
    static final double MAGIC=.008856;
    static final double CBRT_MAGIC = Math.pow(MAGIC, ONE_THIRD);

    public float[] fromCIEXYZ(float[] color) {
	return d2f(fromCIEXYZ(f2d(color)));
    }
    public abstract double[] fromCIEXYZ(double[] color); // my extension

    public float[] toCIEXYZ(float[] color) {
	return d2f(toCIEXYZ(f2d(color)));
    }
    public abstract double[] toCIEXYZ(double[] color); // my extension
    // define sRGB conversion in terms of CIEXYZ conversion.
    public float[] fromRGB(float[] rgb) {
	return d2f(fromRGB(f2d(rgb)));
    }
    public float[] toRGB(float[] colorvalue) {
	return d2f(toRGB(f2d(colorvalue)));
    }
    // use sRGB->CIEXYZ conversion defined in
    //    http://www.w3.org/Graphics/Color/sRGB.html
    public double[] toRGB(double[] colorvalue) {
	double[] xyz = toCIEXYZ(colorvalue);
	//System.out.println("toRGB XYZ:"+d2s(xyz));
	final double x=xyz[0], y=xyz[1], z=xyz[2];
	double r=XYZ2RGB[0][0]*x + XYZ2RGB[0][1]*y + XYZ2RGB[0][2]*z;
	double g=XYZ2RGB[1][0]*x + XYZ2RGB[1][1]*y + XYZ2RGB[1][2]*z;
	double b=XYZ2RGB[2][0]*x + XYZ2RGB[2][1]*y + XYZ2RGB[2][2]*z;
	// clip
	if (r<0) r=0; if (r>1) r=1;
	if (g<0) g=0; if (g>1) g=1;
	if (b<0) b=0; if (b>1) b=1;
	// now gamma correct.
	xyz[0] = (r<=0.00304) ? 12.92*r : (1.055*Math.pow(r,1/2.4)-0.055);
	xyz[1] = (g<=0.00304) ? 12.92*g : (1.055*Math.pow(g,1/2.4)-0.055);
	xyz[2] = (b<=0.00304) ? 12.92*b : (1.055*Math.pow(b,1/2.4)-0.055);
	// done!
	//System.out.println("toRGB RGB:"+d2s(xyz));
	return xyz;
    }
    public double[] fromRGB(double[] rgb) {
	//System.out.println("frRGB RGB:"+d2s(rgb));
	double r=rgb[0], g=rgb[1], b=rgb[2];
	r = (r<=0.03928) ? r/12.92 : Math.pow((r+0.055)/1.055, 2.4);
	g = (g<=0.03928) ? g/12.92 : Math.pow((g+0.055)/1.055, 2.4);
	b = (b<=0.03928) ? b/12.92 : Math.pow((b+0.055)/1.055, 2.4);
	double[] xyz = new double[] {
	    RGB2XYZ[0][0]*r + RGB2XYZ[0][1]*g + RGB2XYZ[0][2]*b,
	    RGB2XYZ[1][0]*r + RGB2XYZ[1][1]*g + RGB2XYZ[1][2]*b,
	    RGB2XYZ[2][0]*r + RGB2XYZ[2][1]*g + RGB2XYZ[2][2]*b,
	};
	//System.out.println("frRGB XYZ:"+d2s(xyz));
	return fromCIEXYZ(xyz);
    }
    // conversion utility functions
    static float[] d2f(double[] d) {
	assert d.length==3;
	return new float[] {(float)d[0],(float)d[1],(float)d[2]};
    }
    static double[] f2d(float[] f) {
	assert f.length==3;
	return new double[] {(double)f[0],(double)f[1],(double)f[2]};
    }
    static String f2s(float[] f) {
	StringBuffer sb = new StringBuffer("{");
	for (int i=0; i<f.length; i++) {
	    sb.append(f[i]);
	    if (i+1<f.length) sb.append(",");
	}
	sb.append('}');
	return sb.toString();
    }
    static String d2s(double[] d) {
	StringBuffer sb = new StringBuffer("{");
	for (int i=0; i<d.length; i++) {
	    sb.append(d[i]);
	    if (i+1<d.length) sb.append(",");
	}
	sb.append('}');
	return sb.toString();
    }
    static final double[][] XYZ2RGB = new double[][] {
	new double[] {  3.2410, -1.5374, -0.4986 },
	new double[] { -0.9692,  1.8760,  0.0416 },
	new double[] {  0.0556, -0.2040,  1.0570 } };
    // compute inverse rather than use the one in the text to ensure
    // round-trip accuracy.
    static final double[][] RGB2XYZ = invert(XYZ2RGB);
    // some brief matrix math to ensure that RGB2XYZ is as accurate as possible
    // doesn't need to be fast, as we only compute this once.
    static double[][] invert(double[][] m) {
	// m[0][0] m[0][1] m[0][2]
	// m[1][0] m[1][1] m[1][2]
	// m[2][0] m[2][1] m[2][2]
	double dm =
	    +m[0][0]*m[1][1]*m[2][2] - m[0][0]*m[1][2]*m[2][1]
	    -m[0][1]*m[1][0]*m[2][2] + m[0][1]*m[1][2]*m[2][0]
	    +m[0][2]*m[1][0]*m[2][1] - m[0][2]*m[1][1]*m[2][0];
	return new double[][] {
	    new double[] { det(m[1][1], m[1][2], m[2][1], m[2][2])/dm,
			   det(m[0][2], m[0][1], m[2][2], m[2][1])/dm,
			   det(m[0][1], m[0][2], m[1][1], m[1][2])/dm },
	    new double[] { det(m[1][2], m[1][0], m[2][2], m[2][0])/dm,
			   det(m[0][0], m[0][2], m[2][0], m[2][2])/dm,
			   det(m[0][2], m[0][0], m[1][2], m[1][0])/dm },
	    new double[] { det(m[1][0], m[1][1], m[2][0], m[2][1])/dm,
			   det(m[0][1], m[0][0], m[2][1], m[2][0])/dm,
			   det(m[0][0], m[0][1], m[1][0], m[1][1])/dm },
	};
    }
    static double det(double m00, double m01, double m10, double m11) {
	return m00*m11 - m01*m10;
    }
}
