package net.cscott.mallard;

import net.cscott.pcbmill.*;
import gnu.getopt.Getopt;
import gnu.getopt.LongOpt;
import java.awt.color.ColorSpace;
import java.awt.image.*;
import java.awt.geom.*;
import java.io.*;
import java.util.AbstractSet;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.*;
import javax.imageio.*;

import net.cscott.jutil.*;

import org.jibble.epsgraphics.EpsGraphics2D;

// for test applet
import java.awt.*;
import javax.swing.*;

import gnu.java.awt.geom.GeneralPath;
import gnu.java.awt.geom.Line2D;

/* This class reads in a delauney triangulation of a polygon.
 * It then:
 *  1) removes interior triangles.
 *  2) traverses midpoints of the triangulation to build a centerline.
 *  3) finds connected components and trims stubs.
 *  4) scales the image.
 *  5) traces it with a 1/16" brush.
 *  6) emits DXF. */
/* XXX: identify bumpy/opal glasses with a special input file containing
 *      one pixel in every "special" region? */
/* XXX: identify "glass minus one" -> which color glass should be purchased
 *      to optimally quantize the image (ie which colors are "missing")
 */
public class Assemble {
    private static final String PROGNAME="Assemble";
    static Rectangle2D SCALEBOX=null;
    static Rectangle2D TRIMBOX=null;
    static Rectangle2D ZOOMBOX=null;
    static double OFFSET=0;
    static double GAPSIZE=0;
    static boolean DO_DISPLAY=false;
    static String FORMAT="dxf";
    static boolean DO_VERBOSE_DISPLAY=false;
    static String colorImageFilename=null;
    static List<StockGlass> glassColors=new ArrayList<StockGlass>();

    public static void main(String[] args)
	throws java.io.IOException,
	       java.awt.geom.NoninvertibleTransformException {

	Getopt g=new Getopt(PROGNAME,args,"hs:t:o:c:i:g:z:f:dW;",new LongOpt[]{
	    new LongOpt("help", LongOpt.NO_ARGUMENT, null, 'h'),
	    new LongOpt("scale", LongOpt.REQUIRED_ARGUMENT, null, 's'),
	    new LongOpt("trim", LongOpt.REQUIRED_ARGUMENT, null, 't'),
	    new LongOpt("offset", LongOpt.REQUIRED_ARGUMENT, null, 'o'),
	    new LongOpt("connect", LongOpt.REQUIRED_ARGUMENT, null, 'c'),
	    new LongOpt("color-image", LongOpt.REQUIRED_ARGUMENT, null, 'i'),
	    new LongOpt("image", LongOpt.REQUIRED_ARGUMENT, null, 'i'),
	    new LongOpt("glass", LongOpt.REQUIRED_ARGUMENT, null, 'g'),
	    new LongOpt("zoom", LongOpt.REQUIRED_ARGUMENT, null, 'z'),
	    new LongOpt("format", LongOpt.REQUIRED_ARGUMENT, null, 'f'),
	    new LongOpt("display", LongOpt.NO_ARGUMENT, null, 'd'),
	});
	int c; String arg; boolean saw_error=false;
	while ((c = g.getopt()) != -1)
	    switch(c) {
	    case 'h':
		saw_error=true; // forces display of help.
		break;
	    case 's':
		// scale image to largest size that will fit inside specified
		// box.
		SCALEBOX = grokbox(g.getOptarg());
		if (SCALEBOX==null) saw_error=true;
		break;
	    case 't':
		// trim the result to the given box. (after scaling)
		TRIMBOX = grokbox(g.getOptarg());
		if (TRIMBOX==null) saw_error=true;
		break;
	    case 'o':
		// trace edges with a marker of the given width.
		OFFSET = Double.parseDouble(g.getOptarg());
		break;
	    case 'c':
		// connect pieces of glass with a gap of the specified width.
		// (note that a gap size of OFFSET is the minimum needed to
		//  get any separation at all.)
		GAPSIZE = Double.parseDouble(g.getOptarg());
		break;
	    case 'i':
		// original color image.
		colorImageFilename = g.getOptarg();
		break;
	    case 'g':
		// specify spectrum glass colors, for quantization.
		glassColors.add(new StockGlass(g.getOptarg()));
		break;
	    case 'f':
		// what type of output is wanted? dxf/png/svg/etc.
		FORMAT=g.getOptarg().toLowerCase();
		break;
	    case 'z':
		// zoom the display.
		ZOOMBOX = grokbox(g.getOptarg());
		if (ZOOMBOX==null) saw_error=true;
		else // flip on y-axis.
		    ZOOMBOX=new Rectangle2D.Double
			(ZOOMBOX.getX(), -ZOOMBOX.getMaxY(),
			 ZOOMBOX.getWidth(), ZOOMBOX.getHeight());
		break;
	    case 'd': // double d gives verbose display.
		if (DO_DISPLAY) DO_VERBOSE_DISPLAY=true;
		DO_DISPLAY = true;
		break;
	    case '?':
		// already printed error msg.
	    default:
		saw_error=true;
		break;
	    }
	if (g.getOptind()+1 != args.length) {
	    System.err.println("Too many arguments.");
	    saw_error=true;
	}
	if (saw_error) {
	    System.err.println("Usage: "+
			       PROGNAME+" [-hs_t_d] <base filename>");
	    System.exit(1);
	}
	String filename = args[g.getOptind()];
	System.err.println("Reading input files.");
	// Read in .node/.poly files (give polygon outline and point locations)
	PolyData pd = readPoly(filename+".1.node", filename+".1.poly");
	// Read in .ele file. (gives triangulation)
	EleData ld = readEle(filename+".1.ele", pd);
	// Read in .edge file. (identifies boundary edges)
	EdgeData ed = readEdge(filename+".1.edge", pd);
	// Read in .neigh file. (triangle neighbors)
	NeighData nd = readNeigh(filename+".1.neigh", ld);

	// Remove interior triangles.
	//  [this is sped up by 'eating away' triangles up to
	//   boundary edges whenever we find an interior/exterior
	//   triangle.  This greatly reduces the number of
	//   triangles tested against the shape.]
	System.err.println("Removing interior triangles.");
	Set<Triangle> interior = new HashSet<Triangle>();
	Set<Triangle> tworkl = new LinkedHashSet<Triangle>(ld.triangles);
	while (!tworkl.isEmpty()) {
	    Triangle tri = tworkl.iterator().next();
	    Set<Triangle> region = eat(tri, ed, nd);
	    assert region.contains(tri);
	    if (!pd.path.contains(tri.centroid()))
		// triangle is in a hole.
		interior.addAll(region);
	    tworkl.removeAll(region);
	}

	ld.triangles.removeAll(interior);
	for (Neighbor n : nd.neighbors.values()) {
	    if (interior.contains(n.t1)) n.t1=null;
	    if (interior.contains(n.t2)) n.t2=null;
	    if (interior.contains(n.t3)) n.t3=null;
	}
	interior=null; tworkl=null; // free memory

	// Build centerline.
	System.err.println("Building centerline.");
	MedialAxis ma = buildCenterline(ld, ed, nd);

	// Analyze connectivity; trim disconnected.
	System.err.println("Finding cycles and trimming.");
	ConnResult cr = findConnected(ma);
	List<Region> regions = cr.regions;
	GeneralPath regionPath = new GeneralPath();
	for (Region r : regions)
	    regionPath.append(r.shape, false);
	// pick out maximal region.
	Region outermost = Collections.max(regions, new Comparator<Region>(){
	    public int compare(Region r1, Region r2) {
		int c1=r1.shape.getBounds2D().contains(r2.shape.getBounds2D())
		?1:0;
		int c2=r2.shape.getBounds2D().contains(r1.shape.getBounds2D())
		?1:0;
		return c1-c2;
	    }
	});

	// Identify colors for each region.
	BufferedImage colorImage=null;
	AffineTransform colorImageXF=null;
	if (colorImageFilename!=null) {
	    System.err.println("Identifying colors.");
	    BufferedImage readImage=ImageIO.read(new File(colorImageFilename));
	    colorImage = new BufferedImage
		(readImage.getWidth(), readImage.getHeight(),
		 BufferedImage.TYPE_INT_ARGB);
	    // image is flipped over y axis compared to everything else.
	    colorImageXF = AffineTransform.getScaleInstance(1,-1);
	    // use polygon outline to set "black" pixels completely
	    // transparent (so they won't throw off the color computation)
	    Graphics2D g2 = colorImage.createGraphics();
	    g2.drawImage(readImage, null, null);//create alpha channel in image
	    g2.setComposite(AlphaComposite.Src);
	    g2.setColor(new Color(0,0,0,0)); // transparent black.
	    g2.transform(colorImageXF.createInverse());
	    g2.fill(pd.path);
	    g2.setStroke(new BasicStroke
			 ((float)(3./colorImageXF.getScaleX()),
			  BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
	    g2.draw(pd.path); // widen the outline a little bit.
	    // okay, now find colors.
	    for (Region r : regions) {
		if (r==outermost) continue; // skip this one.
		System.err.print("."); System.err.flush();
		r.color = findColor(r.shape, colorImage, colorImageXF);
	    }
	    System.err.println();
	}

	// Scale image.
	System.err.println("Scaling path.");
	Rectangle2D bounds = regionPath.getBounds2D();
	AffineTransform at = AffineTransform.getTranslateInstance
	    (-bounds.getMinX(), -bounds.getMinY());
	double scale=1/72.; // 72 dpi default.
	if (SCALEBOX!=null) {
	    double scalex = SCALEBOX.getWidth()/bounds.getWidth();
	    double scaley = SCALEBOX.getHeight()/bounds.getHeight();
	    scale = Math.min(scalex,scaley);
	}
	at.preConcatenate(AffineTransform.getScaleInstance(scale, scale));
	regionPath.transform(at);
	for (Region r: regions)
	    r.transform(at);
	for (RegionBoundary rb: cr.boundaries)
	    rb.transform(at);

	// Connect small pieces.
	if (GAPSIZE>0) {
	    System.err.println("Connecting pieces with gap of "+GAPSIZE+".");
	    List<Region> connectList = new ArrayList<Region>(regions);
	    if (TRIMBOX!=null) {
		// only connect pieces which fall inside the TRIMBOX
		for (Iterator<Region> it=connectList.iterator(); it.hasNext();)
		    if (!it.next().shape.intersects(TRIMBOX))
			it.remove();
	    }
	    regionPath=traceWithGaps(connectList, cr.boundaries,
				     placeGaps(connectList, cr.boundaries));
	}

	// Trace with 1/16" brush.
	Shape stroked;
	if (OFFSET>0) {
	    System.err.println("Offsetting pieces with clearance of "+
			       OFFSET+".");
	    Stroke s = new BasicStroke
		((float)OFFSET, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND);
	    stroked = s.createStrokedShape(regionPath);
	    // shape is a bit buggy:
	    stroked = new Area(stroked);
	} else stroked=regionPath;
	
	// trim to size.
	if (TRIMBOX!=null) {
	    if (!(stroked instanceof Area))
		// converting regionPath to area would result in a solid
		// black square.  need to clip shape on a line-by-line
		// basis, which isn't implemented in the AWT.
		throw new RuntimeException
		    ("Polygon clipping not yet implemented.");
	    System.err.println("Trimming to frame.");
	    Area frame = new Area(TRIMBOX);
	    ((Area)stroked).intersect(frame);
	    ((Area)stroked).exclusiveOr(frame);
	}
	// select glass colors
	if (!glassColors.isEmpty()) {
	    System.err.println("Selecting glass colors.");
	    Set<StockGlass> gs = new HashSet<StockGlass>();
	    for (Region r : regions)
		if (r.color!=null)
		    gs.add(r.glass = selectGlass(glassColors, r.color));
	    System.err.println(gs);
	}
				   
	// Emit as DXF.
	if (FORMAT.equals("dxf")) {
	    System.err.println("Writing DXF.");
	    if (false) { // useful for debugging.
		DXFOutput.writeDXF(new FileWriter(filename+"-tr.dxf"),
				   traceTriangles(ld.triangles), true);
		DXFOutput.writeDXF(new FileWriter(filename+"-ax.dxf"),
				   ma.path, true);
		DXFOutput.writeDXF(new FileWriter(filename+"-cy.dxf"),
				   regionPath, true);
	    }
	    DXFOutput.writeDXF(new FileWriter(filename+".out."+FORMAT),
			       stroked, true);
	}

	// okay, now non-dxf output.
	AffineTransform inv = at.createInverse();
	for (Region r: regions) r.transform(inv);
	stroked = inv.createTransformedShape(stroked);
	final DisplayCtx dc = new DisplayCtx
	    (pd.vertexIndex, pd.path, traceTriangles(ld.triangles),
	     ma.path, stroked, regions, colorImage, colorImageXF);
	if (FORMAT.equals("png") || FORMAT.equals("jpg")) {
	    int w = (ZOOMBOX==null)?768:(int)Math.ceil(ZOOMBOX.getWidth());
	    int h = (ZOOMBOX==null)?768:(int)Math.ceil(ZOOMBOX.getHeight());

	    BufferedImage outImage = new BufferedImage
		(w, h, BufferedImage.TYPE_INT_ARGB);
	    Graphics2D g2 = outImage.createGraphics();
	    g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);
	    doDraw(g2, new Rectangle2D.Double(0,0,w,h), dc);
	    ImageIO.write(outImage, FORMAT, new File(filename+".out."+FORMAT));
	}
	if (FORMAT.equals("svg")) {
	    // scale to fit on a sheet of paper?
	    final double w = (ZOOMBOX!=null) ? ZOOMBOX.getWidth() :
		dc.maxpt.getX()-dc.minpt.getX();
	    final double h = (ZOOMBOX!=null) ? ZOOMBOX.getHeight() :
		dc.maxpt.getY()-dc.minpt.getY();
	    // scale dpi to fit this on a letter-sized sheet of paper.
	    double dpix = w/7.5; // half-inch margins
	    double dpiy = h/10;
	    final int dpi = (int) Math.ceil(Math.max(dpix,dpiy));
	    new SVGOutput() {
		protected Rectangle2D doPaint(Graphics2D g2) {
		    // unscaled bounds
		    doDraw(g2, new Rectangle2D.Double(0,0,w,h), dc);
		    // bounds scaled down by dpi; units are 'inches'
		    return new Rectangle2D.Double(0,0,w/dpi,h/dpi);
		}
	    }.convertShapeAndStream(new FileOutputStream
				    (filename+".out."+FORMAT), dpi);
	}
	if (FORMAT.equals("eps")) {
	    int minx = (int)Math.floor(dc.minpt.getX());
	    int miny = (int)Math.floor(dc.minpt.getY());
	    int maxx = (int)Math.ceil(dc.maxpt.getX());
	    int maxy = (int)Math.ceil(dc.maxpt.getY());
	    Rectangle b = new Rectangle(minx,miny,maxx-minx,maxy-miny);
	    EpsGraphics2D g2 = new EpsGraphics2D
		(filename, new File(filename+".out."+FORMAT),
		 b.x, b.y, b.x+b.width, b.y+b.height);
	    doDraw(g2, b, dc);
	    g2.close();
	}
	if (DO_DISPLAY) display(dc);
    }
    /** Parse a string in the form <float>x<float>+<float>+<float>;
     *  returns a box of the specified width, height, and origin.
     *  If the origin is unspecified, it defaults to 0,0.  If the
     *  height is unspecified, it defaults to the width. */
    private static Rectangle2D grokbox(String s) {
	Matcher m = BOX_PATTERN.matcher(s);
	if (!m.matches()) {
	    System.err.println("Bad box specification: "+s);
	    return null;
	}
	String width=m.group(1);
	String height=m.group(2);
	String x=m.group(3);
	String y=m.group(4);
	if (height==null) height=width;
	if (x==null) x="0";
	if (y==null) y="0";
	return new Rectangle2D.Double
	    (Double.parseDouble(x), Double.parseDouble(y),
	     Double.parseDouble(width), Double.parseDouble(height));
    }
    static final Pattern BOX_PATTERN;
    static {
	String FLOAT="(?:(?:[0-9]+(?:[.][0-9]*)?)|(?:[.][0-9]+))";
	BOX_PATTERN = Pattern.compile
	    ("\\A"+
	     "("+FLOAT+")(?:x("+FLOAT+"))?(?:([+-]"+FLOAT+")([+-]"+FLOAT+")?)?"
	     +"\\Z"
	     );
    }

    // ------------------------------------------------------
    // file i/o routines.

    // read .poly files
    private static class PolyData {
	List<Point2D> vertices;
	Map<Point2D,Integer> vertexIndex;
	List<Line2D> segments;
	GeneralPath path;
    }
    private static void readNodes(TokenReader tr, PolyData pd, int nVertex)
	throws IOException {
	if (nVertex==0) nVertex = tr.readInt(); tr.skipLine();
	pd.vertices = new ArrayList<Point2D>(nVertex);
	pd.vertexIndex = new HashMap<Point2D,Integer>();
	for (int i=0; i<nVertex; i++) {
	    int nPt = tr.readInt(); assert nPt==i;
	    Point2D pt = new Point2D.Double(tr.readDouble(), tr.readDouble());
	    tr.skipLine();
	    pd.vertices.add(pt);
	    pd.vertexIndex.put(pt, pd.vertexIndex.size());
	}
    }
    private static PolyData readPoly(String nodefile, String polyfile)
	throws IOException {
	PolyData pd = new PolyData();
	TokenReader tr = new TokenReader(polyfile);
	// read vertices
	int nVertex = tr.readInt(); tr.skipLine();
	if (nVertex>0)
	    readNodes(tr, pd, nVertex);
	else {
	    TokenReader tr2=new TokenReader(nodefile);
	    readNodes(tr2, pd, 0);
	    tr2.close();
	}
	// read segments
	int nSegment = tr.readInt(); tr.skipLine();
	pd.segments = new ArrayList<Line2D>(nSegment);
	for (int i=0; i<nSegment; i++) {
	    int nPt = tr.readInt(); assert nPt==i;
	    Line2D line = new Line2D.Double
		(pd.vertices.get(tr.readInt()),
		 pd.vertices.get(tr.readInt()));
	    tr.skipLine();
	    pd.segments.add(line);
	}
	// skip rest of file.
	tr.close();
	// build path from segment list.
	//  Treat edges as undirected.
	//  Start with any segment and grow path until we meet the start.
	MultiMap<Point2D,Line2D> endMap = new GenericMultiMap<Point2D,Line2D>
	    (Factories.<Point2D,Collection<Line2D>>linkedHashMapFactory(),
	     new AggregateSetFactory<Line2D>());
	for (Line2D line : pd.segments) {
	    endMap.add(line.getP1(), line);
	    endMap.add(line.getP2(), line);
	}
	pd.path = new GeneralPath(GeneralPath.WIND_EVEN_ODD);
	Point2D first=null, last=null;
	while (!endMap.isEmpty()) {
	    Line2D seg = endMap.values().iterator().next();
	    pd.path.moveTo(seg.getX1(), seg.getY1());
	    first = seg.getP1();
	    last = seg.getP2();
	    endMap.remove(seg.getP1(), seg);
	    endMap.remove(seg.getP2(), seg);
	    while (!first.equals(last)) {
		pd.path.lineTo(last.getX(), last.getY());
		// assume no duplicate edges
		seg = endMap.getValues(last).iterator().next();
		endMap.remove(seg.getP1(), seg);
		endMap.remove(seg.getP2(), seg);
		last = last.equals(seg.getP2()) ? seg.getP1() : seg.getP2();
	    }
	    pd.path.closePath();
	}
	return pd;
    }

    // read .ele files
    static class Triangle {
	final Point2D p1, p2, p3;
	Triangle(Point2D p1, Point2D p2, Point2D p3) {
	    this.p1 = (Point2D) p1.clone();
	    this.p2 = (Point2D) p2.clone();
	    this.p3 = (Point2D) p3.clone();
	}
	Point2D centroid() {
	    return new Point2D.Double((p1.getX()+p2.getX()+p3.getX())/3,
				      (p1.getY()+p2.getY()+p3.getY())/3);
	}
	Shape path() {
	    GeneralPath trianglePath = new GeneralPath();
	    trianglePath.moveTo(p1.getX(), p1.getY());
	    trianglePath.lineTo(p2.getX(), p2.getY());
	    trianglePath.lineTo(p3.getX(), p3.getY());
	    trianglePath.closePath();
	    return trianglePath;
	}
    }
    static GeneralPath traceTriangles(List<Triangle> triangles) {
	GeneralPath trianglePath = new GeneralPath();
	for (Triangle tri : triangles)
	    trianglePath.append(tri.path(), false);
	return trianglePath;
    }

    private static class EleData {
	List<Triangle> triangles;
    }
    private static EleData readEle(String filename, PolyData pd)
	throws IOException {
	EleData ed = new EleData();
	TokenReader tr = new TokenReader(filename);
	// read triangles
	int nTriangle = tr.readInt();
	int nNodes = tr.readInt(); assert nNodes==3;
	tr.skipLine();
	ed.triangles = new ArrayList<Triangle>(nTriangle);
	for (int i=0; i<nTriangle; i++) {
	    int nTr = tr.readInt(); assert nTr==i;
	    Triangle tri = new Triangle(pd.vertices.get(tr.readInt()),
					pd.vertices.get(tr.readInt()),
					pd.vertices.get(tr.readInt()));
	    tr.skipLine();
	    ed.triangles.add(tri);
	}
	// done!
	tr.close();
	return ed;
    }

    // read .edge files
    private static class EdgeData {
	final Set<Line2D> boundaryEdges = new HashSet<Line2D>();
    }
    private static EdgeData readEdge(String filename, PolyData pd)
	throws IOException {
	EdgeData ed = new EdgeData();
	TokenReader tr = new TokenReader(filename);
	// read edges
	int nEdge = tr.readInt();
	int nBoundary = tr.readInt(); assert nBoundary==1;
	tr.skipLine();
	for (int i=0; i<nEdge; i++) {
	    int nEd = tr.readInt(); assert nEd==i;
	    Point2D pt1 = pd.vertices.get(tr.readInt());
	    Point2D pt2 = pd.vertices.get(tr.readInt());
	    int isBoundary = tr.readInt();
	    tr.skipLine();
	    if (isBoundary==1) {
		ed.boundaryEdges.add(new Line2D.Double(pt1, pt2));
		ed.boundaryEdges.add(new Line2D.Double(pt2, pt1));
	    }
	}
	// done!
	tr.close();
	return ed;
    }

    // read .neigh files
    static class Neighbor {
	Triangle t1, t2, t3;
	Neighbor(Triangle t1, Triangle t2, Triangle t3) {
	    this.t1 = t1; this.t2 = t2; this.t3 = t3;
	}
    }
    private static class NeighData {
	final Map<Triangle,Neighbor> neighbors =
	    new LinkedHashMap<Triangle,Neighbor>();
    }
    private static NeighData readNeigh(String filename, EleData ld)
	throws IOException {
	NeighData nd = new NeighData();
	TokenReader tr = new TokenReader(filename);
	// read triangles
	int nTriangle = tr.readInt();
	int nNeigh = tr.readInt(); assert nNeigh==3;
	tr.skipLine();
	for (int i=0; i<nTriangle; i++) {
	    int nTr = tr.readInt(); assert nTr==i;
	    Triangle tri = ld.triangles.get(i);
	    int n1 = tr.readInt(), n2 = tr.readInt(), n3 = tr.readInt();
	    tr.skipLine();
	    Triangle tn1 = (n1==-1)?null:ld.triangles.get(n1);
	    Triangle tn2 = (n2==-1)?null:ld.triangles.get(n2);
	    Triangle tn3 = (n3==-1)?null:ld.triangles.get(n3);
	    Neighbor nei = new Neighbor(tn1, tn2, tn3);
	    nd.neighbors.put(tri, nei);
	}
	// done!
	tr.close();
	return nd;
    }

    /** Eat connected triangles, stopping when a boundary edge is reached. */
    private static Set<Triangle> eat(Triangle tri, EdgeData ed, NeighData nd) {
	Set<Triangle> eaten = new HashSet<Triangle>();
	Set<Triangle> worklist = new LinkedHashSet<Triangle>();
	worklist.add(tri);
	while (!worklist.isEmpty()) {
	    tri = worklist.iterator().next();
	    worklist.remove(tri);

	    eaten.add(tri);
	    Neighbor n = nd.neighbors.get(tri);
	    // three neighbors.
	    // neighbor 1 is adjacent to corners 2 and 3 (opposite corner 1)
	    if (n.t1!=null && !eaten.contains(n.t1) &&
		!ed.boundaryEdges.contains(new Line2D.Double(tri.p2, tri.p3)))
		worklist.add(n.t1);
	    // neighbor 2 is adjacent to corners 3 and 1 (opposite corner 2)
	    if (n.t2!=null && !eaten.contains(n.t2) &&
		!ed.boundaryEdges.contains(new Line2D.Double(tri.p3, tri.p1)))
		worklist.add(n.t2);
	    // neighbor 3 is adjacent to corners 1 and 2 (opposite corner 3)
	    if (n.t3!=null && !eaten.contains(n.t3) &&
		!ed.boundaryEdges.contains(new Line2D.Double(tri.p1, tri.p2)))
		worklist.add(n.t3);
	}
	return eaten;
    }

    // ---------------------------- i/o utilities ------
    static class TokenReader extends BufferedReader {
	TokenReader(String filename) throws IOException {
	    super(new FileReader(filename));
	}
	
	void skipLine() throws IOException { readLine(); }
	int readInt() throws IOException {
	    StringBuffer sb = new StringBuffer();
	    mark(1);
	    int c = read();
	    while (c!=-1 && c!='\n' && c!='\r' &&
		   Character.isWhitespace((char)c)) {
		mark(1);
		c = read();
	    }
	    while (c!=-1 && (c=='-'||c=='+'||Character.isDigit((char)c))) {
		sb.append((char)c);
		mark(1);
		c=read();
	    }
	    if (c<0) throw new RuntimeException("Unexpected EOF");
	    if (!Character.isWhitespace((char)c))
		throw new RuntimeException("Not a digit: "+c);
	    reset();
	    return Integer.parseInt(sb.toString(), 10);
	}
	double readDouble() throws IOException {
	    StringBuffer sb = new StringBuffer();
	    mark(1);
	    int c = read();
	    while (c!=-1 && c!='\n' && c!='\r' &&
		   Character.isWhitespace((char)c)) {
		mark(1);
		c = read();
	    }
	    while (c!=-1 && (c=='.'||c=='-'||c=='+'||c=='e'||c=='E'||
			     Character.isDigit((char)c))) {
		sb.append((char)c);
		mark(1);
		c=read();
	    }
	    if (c<0) throw new RuntimeException("Unexpected EOF");
	    if (!Character.isWhitespace((char)c))
		throw new RuntimeException("Not a digit: "+c);
	    reset();
	    return Double.parseDouble(sb.toString());
	}
    }

    // ----------------------------------------------------------------
    // Approximate the medial axis by tracing through delauney triangles.

    // medial axis transform (approximation)
    static class Node implements Comparable<Node> {
	final Point2D loc;
	final List<Edge> out = new ArrayList<Edge>(3);
	final List<Edge> in = new ArrayList<Edge>(3);
	Node(Point2D loc) { this.loc = loc; }
	public int compareTo(Node n) {
	    // Compares points.
	    // Unfortunately, Point2D doesn't implement Comparable
	    int d = Double.compare(loc.getX(), n.loc.getX());
	    if (d!=0) return d;
	    return Double.compare(loc.getY(), n.loc.getY());
	}
	public int hashCode() {
	    return loc.hashCode();
	}
	public boolean equals(Object o) {
	    if (this==o) return true;
	    if (!(o instanceof Node)) return false;
	    Node n = (Node) o;
	    return loc.equals(n.loc);
	}
	public String toString() { return "N("+loc+")"; }
    }
    static class Edge implements Comparable<Edge> {
	final Node from, to;
	Edge reversed; // reversed partner of this edge.
	Edge(Node from, Node to) {
	    this.from = from; this.to = to;
	}
	public double length() {
	    return from.loc.distance(to.loc);
	}
	public int compareTo(Edge e) {
	    int d = from.compareTo(e.from);
	    if (d!=0) return d;
	    return to.compareTo(e.to);
	}
	public int hashCode(Edge e) {
	    return from.hashCode() + to.hashCode();
	}
	public boolean equals(Object o) {
	    if (this==o) return true;
	    if (!(o instanceof Edge)) return false;
	    Edge e = (Edge) o;
	    return from.equals(e.from) && to.equals(e.to);
	}
	public String toString() { return "E("+from+","+to+")"; }
    }
    static class MedialAxis {
	final Map<Point2D,Node> nodeMap = new LinkedHashMap<Point2D,Node>();
	final List<Edge> allEdges = new ArrayList<Edge>();
	final GeneralPath path = new GeneralPath();

	void addLine(Point2D pt1, Point2D pt2) {
	    Node n1 = pt2node(pt1), n2 = pt2node(pt2);
	    Edge e1 = new Edge(n1, n2);
	    Edge e2 = new Edge(n2, n1);
	    e1.reversed=e2;
	    e2.reversed=e1;
	    n1.out.add(e1);
	    n2.in.add(e1);
	    n1.in.add(e2);
	    n2.out.add(e2);
	    allEdges.add(e1);
	    allEdges.add(e2);
	    // for visualization.
	    path.moveTo(pt1.getX(), pt1.getY());
	    path.lineTo(pt2.getX(), pt2.getY());
	}
	private Node pt2node(Point2D pt) {
	    if (!nodeMap.containsKey(pt)) {
		nodeMap.put(pt, new Node(pt));
	    }
	    return nodeMap.get(pt);
	}
    }
    static Point2D center(Line2D l) {
	return new Point2D.Double((l.getX1()+l.getX2())/2,
				  (l.getY1()+l.getY2())/2);
    }
    static MedialAxis buildCenterline(EleData ld, EdgeData ed, NeighData nd) {
	MedialAxis ma = new MedialAxis();
	// pick a triangle.
	for (Triangle tri : ld.triangles) {
	    // which edges are non-boundaries?
	    Line2D e1 = new Line2D.Double(tri.p1, tri.p2);
	    Line2D e2 = new Line2D.Double(tri.p2, tri.p3);
	    Line2D e3 = new Line2D.Double(tri.p3, tri.p1);
	    List<Line2D> active = new ArrayList<Line2D>(3);
	    if (!ed.boundaryEdges.contains(e1)) active.add(e1);
	    if (!ed.boundaryEdges.contains(e2)) active.add(e2);
	    if (!ed.boundaryEdges.contains(e3)) active.add(e3);
	    // four cases, depending on how many edges are active
	    switch (active.size()) {
	    case 0:
		// nothing to do.
		break;
	    case 1:
		// create a stub edge from active edge to centroid.
		ma.addLine(center(active.get(0)), tri.centroid());
		break;
	    case 2:
		// create an edge between the centers of the active edges.
		ma.addLine(center(active.get(0)), center(active.get(1)));
		break;
	    case 3:
		// create edges between centers and centroid.
		for (Line2D line : active)
		    ma.addLine(center(line), tri.centroid());
		break;
	    default:
		assert false;
	    }
	}
	return ma;
    }

    // ----------------------------------------------------------------
    // find smallest cycles in the medial axis to identify component
    // polygons.

    static class ConnResult {
	final List<Region> regions;
	final Collection<RegionBoundary> boundaries;

	ConnResult(List<Region> regions,
		   Collection<RegionBoundary> boundaries) {
	    this.regions = regions; this.boundaries = boundaries;
	}
    }
    static ConnResult findConnected(MedialAxis ma) {
	Map<Edge,Region> regionMap = new LinkedHashMap<Edge,Region>();
	MultiMap<Region,Edge> regionEdges =
	    new GenericMultiMap(Factories.<Region,Collection<Edge>>
				linkedHashMapFactory(),
				Factories.<Edge>arrayListFactory());
	List<Region> allRegions = new ArrayList<Region>();
	int nCycles=0;

	// algorithm: pick unmarked edge, then search forward always
	// taking the clockwise-most path from a node, until we find
	// a cycle.  mark all the edges in the cycle.  repeat.
	Set<Edge> unmarked = new LinkedHashSet<Edge>(ma.allEdges);
	Set<Edge> invalid = new HashSet<Edge>();
	LinkedList<Edge> cycle = new LinkedList<Edge>();
	Set<Edge> cycleSet = new HashSet<Edge>();
	while (!unmarked.isEmpty()) {
	    Edge start = unmarked.iterator().next();
	    assert cycle.isEmpty(); assert cycleSet.isEmpty();
	    cycle.addFirst(start); cycleSet.add(start);
	    while (!cycle.isEmpty()) {
		Edge head = cycle.getLast();
		Edge tail = findNextCW(head, invalid);
		if (tail==null) { // no valid outgoing edge!
		    invalid.add(head); unmarked.remove(head);
		    invalid.add(head.reversed); unmarked.remove(head.reversed);
		    cycleSet.remove(cycle.removeLast()); // backtrack one.
		} else if (cycleSet.contains(tail)) { // a cycle!
		    // elements at the front are stubs; remove them.
		    while (!cycle.getFirst().equals(tail)) {
			Edge stub = cycle.removeFirst();
			invalid.add(stub); unmarked.remove(stub);
			stub = stub.reversed;
			invalid.add(stub); unmarked.remove(stub);
		    }
		    // okay, now we've got a cycle!
		    unmarked.removeAll(cycle);

		    Region r = new Region(cycle, regionMap);
		    regionEdges.addAll(r, cycle);
		    allRegions.add(r);

		    cycle.clear(); cycleSet.clear();
		    nCycles++;
		    System.err.print("."); System.err.flush();
		} else { // continue this way.
		    assert !cycleSet.contains(tail): tail+" / "+cycle;
		    cycle.addLast(tail); cycleSet.add(tail);
		}
	    }
	}
	System.err.println();
	System.err.println("Found "+nCycles+" cycles.");
	return new ConnResult(allRegions,
			      findBoundaries(regionEdges, regionMap));
    }

    private static class EdgeCWComparator implements Comparator<Edge> {
	final Edge ref;
	EdgeCWComparator(Edge reference) { this.ref = reference; }
	public int compare(Edge e1, Edge e2) {
	    // returns negative if the first is less than the second.
	    assert ref.to.equals(e1.from);
	    assert ref.to.equals(e2.from);
	    int d1= Line2D.relativeCCW(ref.from.loc.getX(),ref.from.loc.getY(),
				       ref.to.loc.getX(),  ref.to.loc.getY(),
				       e1.to.loc.getX(),   e1.to.loc.getY());
	    int d2= Line2D.relativeCCW(ref.from.loc.getX(),ref.from.loc.getY(),
				       ref.to.loc.getX(),  ref.to.loc.getY(),
				       e2.to.loc.getX(),   e2.to.loc.getY());
	    // lowest are clockwise (-1 retval) from in edge.
	    if (d1<d2) return -1;
	    if (d1>d2) return  1;
	    // both on same side of in edge.  return the edge which
	    // is clockwise-most.
	    return Line2D.relativeCCW(e2.from.loc.getX(), e2.from.loc.getY(),
				      e2.to.loc.getX(),   e2.to.loc.getY(),
				      e1.to.loc.getX(),   e1.to.loc.getY());
	}
    }

    static Edge findNextCW(final Edge in, Set<Edge> invalid) {
	Collections.sort(in.to.out, new EdgeCWComparator(in));
	for (Edge e : in.to.out) {
	    if (e.equals(in.reversed)) continue;
	    if (invalid.contains(e)) continue;
	    return e;
	}
	return null; // no valid outgoing edge!
    }

    static List<Edge> reverseCycle(List<Edge> cycle) {
	List<Edge> reversed = new ArrayList<Edge>(cycle.size());
	assert reversed.isEmpty();
	for (Edge e: cycle)
	    reversed.add(e.reversed);
	return reversed;
    }

    static class RegionBoundary {
	final RegionPair adjacent;
	final GeneralPath borderPath = new GeneralPath();
	RegionBoundary(RegionPair adjacent) {
	    this.adjacent=adjacent;
	}
	public String toString() {
	    return "RB["+adjacent+", "+borderPath+", "+length()+"]";
	}
	transient Double cachedLength=null;
	double length() {
	    if (cachedLength==null) {
		double length=0;
		double[] coords = new double[6];
		double startX=0, startY=0, lastX=0, lastY=0;
		for (PathIterator pi = borderPath.getPathIterator(null);
		     !pi.isDone(); pi.next()) {
		    switch(pi.currentSegment(coords)) {
		    case PathIterator.SEG_MOVETO:
			startX = coords[0]; startY = coords[1];
			lastX = coords[0]; lastY = coords[1];
			break;
		    case PathIterator.SEG_CLOSE:
			coords[0] = startX; coords[1] = startY;
			// fall through.
		    case PathIterator.SEG_LINETO:
			// line segment from lastx,lasty to coords[0],coords[1]
			length += Point2D.distance
			    (lastX, lastY, coords[0], coords[1]);
			lastX = coords[0]; lastY = coords[1];
			break;
		    case PathIterator.SEG_QUADTO:
		    case PathIterator.SEG_CUBICTO:
		    default:
			assert false; // should be no curves left.
		    }
		}
		cachedLength=length;
	    }
	    return cachedLength;
	}
	void transform(AffineTransform at) {
	    this.borderPath.transform(at);
	    this.cachedLength=null; // invalidate cached length
	}
    }

    static class Region {
	final GeneralPath shape=new GeneralPath();
	final List<Region> adjacent;
	Color color = null; // possibly-null "mean region color"
	StockGlass glass = null; // possibly-null "selected glass color"
	Region(List<Edge> edges, Map<Edge,Region> regMap) {
	    Set<Region> adjSet = new HashSet<Region>();
	    boolean first=true;
	    int i=0;
	    for (Iterator<Edge> it=edges.iterator(); it.hasNext(); ) {
		Edge e = it.next();
		if (first) {
		    first=false;
		    this.shape.moveTo(e.from.loc.getX(), e.from.loc.getY());
		}
		if (it.hasNext())
		    this.shape.lineTo(e.to.loc.getX(), e.to.loc.getY());
		else
		    this.shape.closePath();
		//
		assert !regMap.containsKey(e);
		regMap.put(e, this);
		if (regMap.containsKey(e.reversed))
		    adjSet.add(regMap.get(e.reversed));
	    }
	    assert !adjSet.contains(this);
	    this.adjacent = new ArrayList<Region>(adjSet);
	    for (Region r: adjacent)
		r.adjacent.add(this);
	}
	void transform(AffineTransform at) {
	    this.shape.transform(at);
	}
    }

    static Collection<RegionBoundary> findBoundaries
	(MultiMap<Region,Edge> regionEdges,
	 Map<Edge,Region> regionMap) {
	List<RegionBoundary> result = new ArrayList<RegionBoundary>();
	Set<Edge> done = new HashSet<Edge>();
	for (Region r: regionEdges.keySet()) {
	    RegionBoundary rb = null;
	    for (Edge e: regionEdges.getValues(r)) {
		// avoid tracing each boundary twice.
		if (done.contains(e)) continue; // skip.
		done.add(e); done.add(e.reversed);
		// find adjacent regions.
		Region r1 = regionMap.get(e);
		Region r2 = regionMap.get(e.reversed);
		assert r1.equals(r) || r2.equals(r);
		RegionPair pair = new RegionPair(r1, r2);
		if (rb==null || !rb.adjacent.equals(pair)) // new boundary!
		    result.add(rb = new RegionBoundary(pair));
		if (rb.borderPath.getCurrentPoint()==null)
		    rb.borderPath.moveTo(e.from.loc.getX(), e.from.loc.getY());
		assert rb.borderPath.getCurrentPoint().equals(e.from.loc);
		rb.borderPath.lineTo(e.to.loc.getX(), e.to.loc.getY());
	    }
	}
	return result;
    }

    static class RegionPair extends TwoSet<Region> {
	RegionPair(Region r1, Region r2) { super(r1, r2); }
    }
    static class TwoSet<T> extends AbstractSet<T> {
	final T left, right;
	TwoSet(T left, T right) { this.left = left; this.right = right; }
	public int size() { return 2; }
	public boolean contains(Object o) { // for efficiency.
	    if (o==null) return left==null || right==null;
	    return o.equals(left) || o.equals(right);
	}
	public Iterator<T> iterator() {
	    return new UnmodifiableIterator<T>() {
		int i=0;
		public T next() {
		    switch(i++) {
		    case 0: return left;
		    case 1: return right;
		    default: i--; throw new java.util.NoSuchElementException();
		    }
		}
		public boolean hasNext() { return i<2; }
	    };
	}
    }

    // ----------------------------------------------------------------
    // find region colors
    static Color findColor(BufferedImage colorImage) {
	try {
	    return findColor(null, colorImage, new AffineTransform());
	} catch (java.awt.geom.NoninvertibleTransformException e) {
	    assert false; // impossible!
	    return null;
	}
    }
    static Color findColor(Shape s,
			   BufferedImage colorImage,
			   AffineTransform colorImageXF)
	throws java.awt.geom.NoninvertibleTransformException {
	AffineTransform inv = colorImageXF.createInverse();
	int minX, minY, maxX, maxY; BufferedImage mask;
	if (s==null) {
	    mask=null; minX=minY=0;
	    maxX=colorImage.getWidth()-1; maxY=colorImage.getHeight()-1;
	} else {
	    Shape sT = inv.createTransformedShape(s);
	    // compute bounding box of region, transform to coord.
	    // space of color image, and then for each pixel in that region:
	    Rectangle2D bounds = sT.getBounds2D();
	    minX=(int)Math.round(Math.ceil(bounds.getMinX()));
	    minY=(int)Math.round(Math.ceil(bounds.getMinY()));
	    maxX=(int)Math.round(Math.floor(bounds.getMaxX()));
	    maxY=(int)Math.round(Math.floor(bounds.getMaxY()));

	    // make the shape into a mask image.
	    mask = new BufferedImage
		(1+maxX-minX, 1+maxY-minY, BufferedImage.TYPE_BYTE_BINARY);
	    Graphics2D g2 = mask.createGraphics();
	    g2.setColor(Color.WHITE);
	    g2.fill(new Rectangle2D.Double(0,0,1+maxX-minX,1+maxY-minY));
	    g2.transform(AffineTransform.getTranslateInstance(-minX, -minY));
	    g2.transform(inv);
	    g2.setColor(Color.BLACK);
	    g2.fill(s);
	}
	int stride = maxX-minX+1;

	Point2D p=new Point2D.Double(), pT=new Point2D.Double();

	double avgR=0, avgG=0, avgB=0, sum=0;
	int[] pixels = colorImage.getRGB(minX, minY, 1+maxX-minX, 1+maxY-minY,
					 null, 0, stride);
	for (int x=minX; x<=maxX; x++)
	    for (int y=minY; y<=maxY; y++) {
		p.setLocation((double)x,(double)y); inv.transform(p, pT);
		//   check to see if pixel is in the region. (if not, skip)
		if (mask!=null &&
		    0!=(mask.getRGB(x-minX, y-minY)&0xFFFFFF)) continue;
		//   compute average color
		int pixel = pixels[(y-minY)*stride + (x-minX)];
		double a = ((pixel>>24)&0xFF)/255.;
		double r = ((pixel>>16)&0xFF)/255.;
		double g = ((pixel>> 8)&0xFF)/255.;
		double b = ((pixel    )&0xFF)/255.;
		// remove gamma correction to restore linear space.
		r = (r<=0.03928) ? r/12.92 : Math.pow((r+0.055)/1.055, 2.4);
		g = (g<=0.03928) ? g/12.92 : Math.pow((g+0.055)/1.055, 2.4);
		b = (b<=0.03928) ? b/12.92 : Math.pow((b+0.055)/1.055, 2.4);
		// okay, add to weighted average.
		final double W=a;
		avgR+=W*r;
		avgG+=W*g;
		avgB+=W*b;
		sum+=W;
	    }
	if (sum==0) sum=1; // avoid divide-by-zero.
	double r = avgR/sum;
	double g = avgG/sum;
	double b = avgB/sum;
	// restore gamma correction.
	r = (r<=0.00304) ? 12.92*r : (1.055*Math.pow(r,1/2.4)-0.055);
	g = (g<=0.00304) ? 12.92*g : (1.055*Math.pow(g,1/2.4)-0.055);
	b = (b<=0.00304) ? 12.92*b : (1.055*Math.pow(b,1/2.4)-0.055);
	// return that color!
	return new Color((float)r, (float)g, (float)b, 1f);
    }
    static Point2D findCentroid(Shape s) {
	// do computations relative to center of bounding box, for greater
	// accuracy/lesser chance of overflow.
	Rectangle2D bounds = s.getBounds2D();
	double originX = bounds.getCenterX();
	double originY = bounds.getCenterY();
	
	double area=0, centroidX=0, centroidY=0;

	double[] coords = new double[6];
	double startX=0, startY=0, lastX=0, lastY=0;
	for (PathIterator pi = s.getPathIterator(null);
	     !pi.isDone(); pi.next()) {
	    switch(pi.currentSegment(coords)) {
	    case PathIterator.SEG_MOVETO:
		startX = coords[0]; startY = coords[1];
		lastX = coords[0]; lastY = coords[1];
		break;
	    case PathIterator.SEG_CLOSE:
		coords[0] = startX; coords[1] = startY;
		// fall through.
	    case PathIterator.SEG_LINETO:
		// line segment from lastx,lasty to coords[0],coords[1]
		double term = (lastX-originX)*(coords[1]-originY)
		    - (coords[0]-originX)*(lastY-originY);
		area += term;
		centroidX+=(lastX+coords[0]-2*originX)*term;
		centroidY+=(lastY+coords[1]-2*originY)*term;

		lastX = coords[0]; lastY = coords[1];
		break;
	    case PathIterator.SEG_QUADTO:
	    case PathIterator.SEG_CUBICTO:
	    default:
		assert false; // should be no curves left.
	    }
	}
	area /= 2;
	centroidX /= 6*area;
	centroidY /= 6*area;
	return new Point2D.Double(centroidX+originX, centroidY+originY);
    }

    // ----------------------------------------------------------------
    // connect small pieces with sprue
    static GeneralPath traceWithGaps(List<Region> connectList,
				     Collection<RegionBoundary> allBoundaries,
				     Set<RegionBoundary> gappedBoundaries) {
	Set<Region> connectSet = new HashSet<Region>(connectList);

	GeneralPath result = new GeneralPath();
	double[] coords = new double[6];
	double startx=0, starty=0, lastx=0, lasty=0;
	Point2D needMove = null;

	for (RegionBoundary rb : allBoundaries) {
	    if (!(connectSet.contains(rb.adjacent.left) ||
		  connectSet.contains(rb.adjacent.right)))
		continue; // skip this boundary.
	    double len = rb.length();
	    double startGap, endGap;
	    if (gappedBoundaries.contains(rb)) {
		startGap=(len-GAPSIZE)/2;
		endGap=(len+GAPSIZE)/2;
	    } else startGap=endGap=Double.POSITIVE_INFINITY;
	    
	    len=0;
	    for (PathIterator pi = rb.borderPath.getPathIterator(null);
		 !pi.isDone(); pi.next())
		switch(pi.currentSegment(coords)) {
		case PathIterator.SEG_MOVETO:
		    startx = coords[0]; starty = coords[1];
		    lastx = coords[0]; lasty = coords[1];
		    needMove = new Point2D.Double(startx, starty);
		    break;
		case PathIterator.SEG_CLOSE:
		    coords[0] = startx; coords[1] = starty;
		    // fall through.
		case PathIterator.SEG_LINETO:
		    // figure out pathlength to start and end of segment.
		    double seglen = Point2D.distance
			(lastx, lasty, coords[0], coords[1]);
		    // if line falls completely outside gap, draw it.
		    if (len+seglen <= startGap || len > endGap) {
			needMove = doLineTo(result, needMove,
					    coords[0], coords[1]);
		    } else {
			// if line straddles startGap, then draw first part.
			if (len <= startGap && len+seglen > startGap) {
			    double scale = (startGap-len)/seglen;
			    needMove=doLineTo(result, needMove,
					      scale*coords[0]+(1-scale)*lastx,
					      scale*coords[1]+(1-scale)*lasty);
			}
			// if line straddles endGap, then draw last part.
			if (len <= endGap && len+seglen > endGap) {
			    double scale = (endGap-len)/seglen;
			    needMove = new Point2D.Double
				(scale*coords[0]+(1-scale)*lastx,
				 scale*coords[1]+(1-scale)*lasty);
			    needMove=doLineTo(result, needMove,
					      coords[0], coords[1]);
			}
		    }
		    lastx = coords[0]; lasty = coords[1];
		    len += seglen;
		    break;
		default:
		    assert false;
		}
	}
	return result;
    }
    private static Point2D doLineTo(GeneralPath result,Point2D needMove,
				    double x, double y) {
	// this business with needMove is necessary because BasicStroke
	// pitches a fit if you have adjacent moveTo operations in your
	// path.  It always assumes the result of a moveTo is an endpoint,
	// and so the result appears as if you had zero-length segments
	// at every moveTo location (as opposed to moving without leaving
	// a mark, which is what you'd probably expect).
	if (needMove!=null) result.moveTo(needMove.getX(), needMove.getY());
	result.lineTo(x, y);
	return null;
    }

    static Set<RegionBoundary> placeGaps
	(List<Region> connectList, Collection<RegionBoundary> boundaries) {
	Set<RegionBoundary> gappedBoundaries = new HashSet<RegionBoundary>();
	// we will pick edges in order from longest to shortest, adding a
	// gap iff the edge connects regions which are not already connected.
	// this will guarantee that we pick the longest edges sufficient to
	// join all the pieces.
	Set<Region> connectSet = new HashSet<Region>(connectList);
	List<RegionBoundary> candidates = new ArrayList<RegionBoundary>();
	for (RegionBoundary rb: boundaries)
	    if (connectSet.contains(rb.adjacent.left) &&
		connectSet.contains(rb.adjacent.right))
		candidates.add(rb);
	Collections.sort(candidates, new Comparator<RegionBoundary>() {
	    public int compare(RegionBoundary rb1, RegionBoundary rb2) {
		return -Double.compare(rb1.length(), rb2.length());
	    }
	});
	DisjointSet<Region> groups = new DisjointSet<Region>();
	for (RegionBoundary rb : candidates) {
	    if (groups.find(rb.adjacent.left)==groups.find(rb.adjacent.right))
		continue; // already joined.  don't make a cycle!
	    // add edge between rb.adjacent regions
	    groups.union(rb.adjacent.left, rb.adjacent.right);
	    gappedBoundaries.add(rb);
	}
	// done!
	return gappedBoundaries;
    }

    // --------------------------------------------------------------
    // fetch information about stock glass colors.
    static class StockGlass {
	final String name;
	final BufferedImage image;
	final Color color;
	StockGlass(String name) throws IOException {
	    this.name = name;
	    String url =
		"http://www.warner-criv.com/images/products/medium/"+
		canonicalize(name)+".jpg";
	    this.image = ImageIO.read(new java.net.URL(url));
	    if (this.image==null)
		throw new IOException("Glass color image not found: "+name);
	    this.color = findColor(this.image);
	}
	static String canonicalize(String glassName) {
	    // remove all non-alphanum characters, and then insert a dash
	    // after the first four numbers.
	    String canon =
		NONALNUM_PATTERN.matcher(glassName)
		.replaceAll("").toUpperCase();
	    if (canon.length()>4)
		canon = canon.substring(0, 4)+"-"+canon.substring(4);
	    return canon;
	}
	static final Pattern NONALNUM_PATTERN =
	    Pattern.compile("[^\\p{Alnum}]+");
	public int hashCode() {
	    return canonicalize(name).hashCode();
	}
	public boolean equals(Object o) {
	    if (!(o instanceof StockGlass)) return false;
	    StockGlass g = (StockGlass) o;
	    return canonicalize(name).equals(canonicalize(g.name));
	}
	public String toString() {
	    return name;
	}
    }
    //
    static StockGlass selectGlass(Collection<StockGlass> glassList,
				  final Color c) {
	return Collections.min(glassList, new Comparator<StockGlass>() {
	    public int compare(StockGlass g1, StockGlass g2) {
		return Float.compare(colorDiff(c, g1.color),
				     colorDiff(c, g2.color));
	    }
	});
    }
    static final ColorSpace CS_Luv = LuvColorSpace.getInstance();
    static float colorDiff(Color c1, Color c2) {
	// use euclidean distance-squared between colors in L*u*v*
	float[] f1 = CS_Luv.fromRGB(c1.getRGBColorComponents(null));
	float[] f2 = CS_Luv.fromRGB(c2.getRGBColorComponents(null));
	assert f1.length==3; assert f2.length==3;
	float d0 = f1[0]-f2[0], d1 = f1[1]-f2[1], d2 = f1[2]-f2[2];
	// weight luminance half as much as color difference.
	return d0*d0/4 + d1*d1 + d2*d2;
    }

    // --------------------------------------------------------------
    // testing: display various paths on the screen.
    static void display(DisplayCtx dc) {
        JFrame f = new JFrame(PROGNAME);
        f.addWindowListener(new java.awt.event.WindowAdapter() {
		public void windowClosing(java.awt.event.WindowEvent e) {
		    System.exit(0);
		}
	    });
        JApplet applet = new App(dc);
        f.getContentPane().add("Center", applet);
        applet.init();
        f.pack();
        f.setSize(new Dimension(700,700));
        f.show();	
    }
    static class DisplayCtx {
	final Map<Point2D,Integer> contourIndex;
	final Point2D minpt, maxpt;
	final Shape contourPath, trianglePath, axisPath, cyclePath;
	final BufferedImage colorImage;
	final AffineTransform colorImageXF;
	final List<Region> regions;
	final boolean DRAW_DOTS=DO_VERBOSE_DISPLAY;
	final boolean DRAW_POLY=DO_VERBOSE_DISPLAY;
	final boolean DRAW_TRI=DO_VERBOSE_DISPLAY;
	final boolean DRAW_AXIS=true;
	final boolean DRAW_CYCLES=true;
	final boolean DRAW_COLOR_IMAGE=DO_VERBOSE_DISPLAY;
	final boolean DRAW_COLOR_REGIONS=true;
	final boolean QUANTIZE_COLORS=true;
	DisplayCtx(Map<Point2D,Integer> contourIndex,
		   Shape contourPath,
		   Shape trianglePath,
		   Shape axisPath,
		   Shape cyclePath,
		   List<Region> regions,
		   BufferedImage colorImage, AffineTransform colorImageXF) {
	    this.contourIndex = contourIndex;
	    this.contourPath = contourPath;
	    if (ZOOMBOX==null) {
		double minx=Double.POSITIVE_INFINITY;
		double miny=Double.POSITIVE_INFINITY;
		double maxx=Double.NEGATIVE_INFINITY;
		double maxy=Double.NEGATIVE_INFINITY;
		for (Point2D pt : contourIndex.keySet()) {
		    if (pt.getX() < minx) minx = pt.getX();
		    if (pt.getY() < miny) miny = pt.getY();
		    if (pt.getX() > maxx) maxx = pt.getX();
		    if (pt.getY() > maxy) maxy = pt.getY();
		}
		this.minpt = new Point2D.Double(minx, miny);
		this.maxpt = new Point2D.Double(maxx, maxy);
	    } else {
		this.minpt = new Point2D.Double
		    (ZOOMBOX.getMinX(), ZOOMBOX.getMinY());
		this.maxpt = new Point2D.Double
		    (ZOOMBOX.getMaxX(), ZOOMBOX.getMaxY());
	    }
	    this.trianglePath = trianglePath;
	    this.axisPath = axisPath;
	    this.cyclePath = cyclePath;
	    this.regions = regions;
	    this.colorImage = colorImage;
	    this.colorImageXF = colorImageXF;
	}
    }
    static class App extends javax.swing.JApplet {
	final static Color bg = Color.white;
	final static Color fg = Color.black;
	final DisplayCtx dc;

	App(DisplayCtx dc) { this.dc = dc; }

	public void init() {
	    //Initialize drawing colors
	    setBackground(bg);
	    setForeground(fg);
	}
	public void paint(Graphics g) {
	    Graphics2D g2 = (Graphics2D) g;
	    g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);
	    Dimension d = getSize();
	    g2.setPaint(fg);

	    // here's where the real work is done.
	    doDraw(g2, new Rectangle2D.Double(8,8,d.width-16,d.height-16), dc);
	
	    Color fg3D = Color.lightGray;
	    g2.setPaint(fg3D);
	    g2.draw3DRect(0, 0, d.width - 1, d.height - 1, true);
	    g2.draw3DRect(3, 3, d.width - 7, d.height - 7, false);
	    g2.setPaint(fg);
	}
    }
    static void doDraw(Graphics2D g2, Rectangle2D bounds, DisplayCtx dc) {
	g2.setStroke(new BasicStroke());
	// create area scaled to fit.
	// also flip the y axis, since gerber has the origin at the lower left
	AffineTransform t = AffineTransform.getTranslateInstance
	    (-dc.minpt.getX(), -dc.minpt.getY());
	
	t.preConcatenate
	    (AffineTransform.getScaleInstance(1, -1));
	t.preConcatenate
	    (AffineTransform.getTranslateInstance
	     (0, dc.maxpt.getY()-dc.minpt.getY()));
	
	double scaleX=bounds.getWidth()/(dc.maxpt.getX()-dc.minpt.getX());
	double scaleY=bounds.getHeight()/(dc.maxpt.getY()-dc.minpt.getY());
	double scale = Math.min(scaleX, scaleY);
	t.preConcatenate
	    (AffineTransform.getScaleInstance(scale,scale));
	t.preConcatenate
	    (AffineTransform.getTranslateInstance
	     (bounds.getMinX(),bounds.getMinY()));

	if (dc.DRAW_COLOR_IMAGE && dc.colorImage!=null) {
	    AffineTransform at = (AffineTransform) t.clone();
	    at.concatenate(dc.colorImageXF);
	    g2.drawImage(dc.colorImage, at, null);
	}
	
	if (dc.DRAW_COLOR_REGIONS)
	    for (Region r : dc.regions)
		if (dc.QUANTIZE_COLORS && r.glass!=null) {
			StockGlass sg = r.glass;
			Shape rT=t.createTransformedShape(r.shape);
			Rectangle2D b = rT.getBounds2D();
			double sx = b.getWidth()/sg.image.getWidth();
			double sy = b.getHeight()/sg.image.getHeight();
			double s = Math.max(sx,sy);
			AffineTransform at =
			    AffineTransform.getScaleInstance(s,s);
			at.preConcatenate
			    (AffineTransform.getTranslateInstance
			     (b.getMinX(), b.getMinY()));
			Shape oldClip = g2.getClip();
			g2.clip(rT);
			g2.drawImage(sg.image, at, null);
			g2.setClip(oldClip);
		} else if (r.color!=null) {
			g2.setPaint(r.color);
			g2.fill(t.createTransformedShape(r.shape));
		}
	
	if (dc.DRAW_DOTS) {
	    Shape dot = new Ellipse2D.Double(-2,-2,4,4);
	    for (Point2D pt : dc.contourIndex.keySet()) {
		float percent =
		    dc.contourIndex.get(pt).floatValue() /
		    dc.contourIndex.size();
		g2.setPaint(Color.getHSBColor(percent,1,0.75f));
		Point2D loc = t.transform(pt, null);
		g2.fill(AffineTransform.getTranslateInstance
			(loc.getX(),
			 loc.getY()).createTransformedShape(dot));
	    }
	}

	if (dc.DRAW_POLY) {
	    g2.setPaint(Color.red);
	    g2.draw(t.createTransformedShape(dc.contourPath));
	}

	if (dc.DRAW_TRI) {
	    g2.setPaint(Color.blue);
	    g2.draw(t.createTransformedShape(dc.trianglePath));
	}

	if (dc.DRAW_AXIS) {
	    g2.setPaint(Color.green);
	    g2.draw(t.createTransformedShape(dc.axisPath));
	}

	if (dc.DRAW_CYCLES) {
	    g2.setPaint(Color.orange);
	    g2.draw(t.createTransformedShape(dc.cyclePath));
	}
    }
}
